/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.audio;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IObjectArray;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.services.IAudioEffectContext;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.StereoMixer", category=Effect.AUDIO)
public class StereoMixer {

	@Property(value="100", min="0", max="400")
	private IAnimatableDouble leftLevel;

	@Property(value="100", min="0", max="400")
	private IAnimatableDouble rightLevel;

	@Property(value="-100", min="-100", max="100")
	private IAnimatableDouble leftPan;

	@Property(value="100", min="-100", max="100")
	private IAnimatableDouble rightPan;

	@Property
	private IAnimatableBoolean invertPhase;


	private final IAudioEffectContext context;

	@Inject
	public StereoMixer(IAudioEffectContext context) {
		this.context = context;
	}

	public IAudioBuffer doAudioEffect() {
		IObjectArray<Double> leftLevels = context.values(leftLevel);
		IObjectArray<Double> rightLevels = context.values(rightLevel);
		IObjectArray<Double> leftPans = context.values(leftPan);
		IObjectArray<Double> rightPans = context.values(rightPan);
		IObjectArray<Boolean> phases = context.values(invertPhase);

		IAudioBuffer ab = context.doPreviousEffect();

		Object data = ab.getData();
		int frameCount = ab.getFrameCount();

		switch (ab.getAudioMode().dataType) {
			case SHORT:
				mixer((short[]) data, frameCount, leftLevels, rightLevels, leftPans, rightPans, phases);
				break;

			case INT:
				mixer((int[]) data, frameCount, leftLevels, rightLevels, leftPans, rightPans, phases);
				break;

			case FLOAT:
				mixer((float[]) data, frameCount, leftLevels, rightLevels, leftPans, rightPans, phases);
				break;
		}

		leftLevels.release();
		rightLevels.release();
		leftPans.release();
		rightPans.release();
		phases.release();

		return ab;
	}

	private void mixer(short[] data, int frameCount,
			IObjectArray<Double> leftLevels, IObjectArray<Double> rightLevels,
			IObjectArray<Double> leftPans, IObjectArray<Double> rightPans,
			IObjectArray<Boolean> phases) {

		Object[] leftLevelsArray = leftLevels.getArray();
		Object[] rightLevelsArray = rightLevels.getArray();
		Object[] leftPansArray = leftPans.getArray();
		Object[] rightPansArray = rightPans.getArray();
		Object[] phasesArray = phases.getArray();

		for (int i = 0; i < frameCount; ++i) {
			double leftLevel = (Double)leftLevelsArray[i] / 100.0;				// [0, 400] -> [0, 4]
			double rightLevel = (Double)rightLevelsArray[i] / 100.0;
			double leftPan = ((Double)leftPansArray[i] - 100.0) / -200.0;		// [-100,  100] -> [1, 0]
			double rightPan = ((Double)rightPansArray[i] + 100.0) / 200.0;		// [ 100, -100] -> [1, 0]
			double phase = (Boolean)phasesArray[i] ? -1.0 : 1.0;

			double left  = data[i*2  ] * leftLevel;
			double right = data[i*2+1] * rightLevel;
			data[i*2  ] = clampToShort((left  * leftPan  + right * (1.0 - rightPan)) * phase);
			data[i*2+1] = clampToShort((right * rightPan + left  * (1.0 - leftPan )) * phase);
		}
	}

	private void mixer(int[] data, int frameCount,
			IObjectArray<Double> leftLevels, IObjectArray<Double> rightLevels,
			IObjectArray<Double> leftPans, IObjectArray<Double> rightPans,
			IObjectArray<Boolean> phases) {

		Object[] leftLevelsArray = leftLevels.getArray();
		Object[] rightLevelsArray = rightLevels.getArray();
		Object[] leftPansArray = leftPans.getArray();
		Object[] rightPansArray = rightPans.getArray();
		Object[] phasesArray = phases.getArray();

		for (int i = 0; i < frameCount; ++i) {
			double leftLevel = (Double)leftLevelsArray[i] / 100.0;
			double rightLevel = (Double)rightLevelsArray[i] / 100.0;
			double leftPan = ((Double)leftPansArray[i] - 100.0) / -200.0;
			double rightPan = ((Double)rightPansArray[i] + 100.0) / 200.0;
			double phase = (Boolean)phasesArray[i] ? -1.0 : 1.0;

			double left  = data[i*2  ] * leftLevel;
			double right = data[i*2+1] * rightLevel;
			data[i*2  ] = clampToInt((left  * leftPan  + right * (1.0 - rightPan)) * phase);
			data[i*2+1] = clampToInt((right * rightPan + left  * (1.0 - leftPan )) * phase);
		}
	}

	private void mixer(float[] data, int frameCount,
			IObjectArray<Double> leftLevels, IObjectArray<Double> rightLevels,
			IObjectArray<Double> leftPans, IObjectArray<Double> rightPans,
			IObjectArray<Boolean> phases) {

		Object[] leftLevelsArray = leftLevels.getArray();
		Object[] rightLevelsArray = rightLevels.getArray();
		Object[] leftPansArray = leftPans.getArray();
		Object[] rightPansArray = rightPans.getArray();
		Object[] phasesArray = phases.getArray();

		for (int i = 0; i < frameCount; ++i) {
			double leftLevel = (Double)leftLevelsArray[i] / 100.0;
			double rightLevel = (Double)rightLevelsArray[i] / 100.0;
			double leftPan = ((Double)leftPansArray[i] - 100.0) / -200.0;
			double rightPan = ((Double)rightPansArray[i] + 100.0) / 200.0;
			double phase = (Boolean)phasesArray[i] ? -1.0 : 1.0;

			double left  = data[i*2  ] * leftLevel;
			double right = data[i*2+1] * rightLevel;
			data[i*2  ] = (float) ((left  * leftPan  + right * (1.0 - rightPan)) * phase);
			data[i*2+1] = (float) ((right * rightPan + left  * (1.0 - leftPan )) * phase);
		}
	}

	private short clampToShort(double d) {
		return (short) Math.min(Math.max(d, Short.MIN_VALUE), Short.MAX_VALUE); 
	}

	private int clampToInt(double d) {
		return (int) Math.min(Math.max(d, Integer.MIN_VALUE), Integer.MAX_VALUE); 
	}

}
