/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.blurSharpen;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.effects.VideoEffectUtil;

public class BlurUtil {

	private static final int[][] DOWNSAMPLE_FACTORS = {
		{ 300, 7, 7 },
		{ 200, 5, 7 },
		{ 150, 5, 5 },
		{ 100, 3, 5 },
		{  50, 3, 3 },
		{  30, 5 },
		{  10, 3 }
	};

	public static int[] getDownSampleFactors(double hint) {
		for (int i = 0; i < DOWNSAMPLE_FACTORS.length; ++i) {
			if (hint >= DOWNSAMPLE_FACTORS[i][0]) {
				int[] factors = new int[DOWNSAMPLE_FACTORS[i].length - 1];
				for (int j = 0; j < factors.length; ++j) {
					factors[j] = DOWNSAMPLE_FACTORS[i][j+1];
				}
				return factors;
			}
		}
		return new int[0];
	}

	public static void doDownSample(
			IVideoBuffer src, IVideoBuffer dst, int hRatio, int vRatio,
			final GL2 gl, GLU glu, IShaderRegistry shaders) {

		final int ksize = hRatio * vRatio;
		final float[] kernel = new float[ksize];
		final float[] offset = new float[ksize*2];

		for (int j = 0; j < vRatio; ++j) {
			for (int i = 0; i < hRatio; ++i) {
				int k = j*hRatio+i;
				kernel[k] = 1f/ksize;
				offset[k*2  ] = i - hRatio/2;
				offset[k*2+1] = j - vRatio/2;
			}
		}

		VideoBounds dstBounds = dst.getBounds();

		final float quadWidth = dstBounds.width * hRatio;
		final float quadHeight = dstBounds.height * vRatio;

		VideoEffectUtil.ortho2D(gl, glu, dstBounds.width, dstBounds.height);
		gl.glScalef(1f/hRatio, 1f/vRatio, 1);

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, dst.getTexture(), 0);
		gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

		gl.glActiveTexture(GL2.GL_TEXTURE0);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, src.getTexture());

		final IShaderProgram program = shaders.getProgram(VideoEffectUtil.class, "CONVOLUTION");
		program.useProgram(new Runnable() {
			public void run() {
				gl.glUniform1i(program.getUniformLocation("texture"), 0);
				gl.glUniform1i(program.getUniformLocation("ksize"), ksize);
				gl.glUniform1fv(program.getUniformLocation("kernel[0]"), ksize, kernel, 0);
				gl.glUniform2fv(program.getUniformLocation("offset[0]"), ksize, offset, 0);

				gl.glBegin(GL2.GL_QUADS);
				gl.glTexCoord2f(0, 0);
				gl.glVertex2f(0, 0);
				gl.glTexCoord2f(quadWidth, 0);
				gl.glVertex2f(quadWidth, 0);
				gl.glTexCoord2f(quadWidth, quadHeight);
				gl.glVertex2f(quadWidth, quadHeight);
				gl.glTexCoord2f(0, quadHeight);
				gl.glVertex2f(0, quadHeight);
				gl.glEnd();
			}
		});

		gl.glActiveTexture(GL2.GL_TEXTURE0);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
	}

	public static void doUpSample(
			IVideoBuffer src, IVideoBuffer dst,
			int hRatio, int vRatio, GL2 gl, GLU glu) {

		VideoBounds srcBounds = src.getBounds();
		VideoBounds dstBounds = dst.getBounds();

		VideoEffectUtil.ortho2D(gl, glu, dstBounds.width, dstBounds.height);
		gl.glScalef(hRatio, vRatio, 1);

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, dst.getTexture(), 0);
		gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

		gl.glActiveTexture(GL2.GL_TEXTURE0);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, src.getTexture());
		gl.glEnable(GL2.GL_TEXTURE_RECTANGLE);

		gl.glColor4f(1, 1, 1, 1);

		gl.glBegin(GL2.GL_QUADS);
		gl.glTexCoord2f(0, 0);
		gl.glVertex2f(0, 0);
		gl.glTexCoord2f(srcBounds.width, 0);
		gl.glVertex2f(srcBounds.width, 0);
		gl.glTexCoord2f(srcBounds.width, srcBounds.height);
		gl.glVertex2f(srcBounds.width, srcBounds.height);
		gl.glTexCoord2f(0, srcBounds.height);
		gl.glVertex2f(0, srcBounds.height);
		gl.glEnd();

		gl.glActiveTexture(GL2.GL_TEXTURE0);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
		gl.glDisable(GL2.GL_TEXTURE_RECTANGLE);

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
	}

}
