/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.colorCorrection;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.BrightnessAndContrast", category=Effect.COLOR_CORRECTION)
public class BrightnessAndContrast {

	@ShaderSource
	public static final String[] BRIGHTNESS_AND_CONTRAST = {
		"uniform sampler2DRect texture;",
		"uniform float brightness;",
		"uniform float contrastSlope;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
		"	vec3 rgb = color.rgb/color.a;",
		"	rgb = (rgb+brightness-0.5)*contrastSlope+0.5;",
		"	gl_FragColor = vec4(rgb*color.a, color.a);",
		"}"
	};


	@Property(min="-255", max="255")
	private IAnimatableDouble brightness;

	@Property(min="-100", max="100")
	private IAnimatableDouble contrast;


	private final IVideoEffectContext context;

	private final IShaderProgram program;

	@Inject
	public BrightnessAndContrast(IVideoEffectContext context, IShaderRegistry shaders) {
		this.context = context;
		program = shaders.getProgram(BrightnessAndContrast.class, "BRIGHTNESS_AND_CONTRAST");
	}

	public IVideoBuffer doVideoEffect() {
		final double brightness = context.value(this.brightness);
		double contrast = context.value(this.contrast);

		if (brightness == 0 && contrast == 0) {
			return null;
		}

		final double contrastSlope;
		if (contrast >= 0) {
			contrastSlope = 100/(100-contrast);
		} else {
			contrastSlope = (100+contrast)/100;
		}

		IVideoBuffer input = null;
		IVideoBuffer buffer = null;
		try {
			input = context.doPreviousEffect();
			VideoBounds bounds = input.getBounds();
			if (bounds.isEmpty()) {
				IVideoBuffer output = input;
				input = null;
				return output;
			}

			final GL2 gl = context.getGL().getGL2();

			gl.glPushAttrib(GL2.GL_TEXTURE_BIT | GL2.GL_CURRENT_BIT);
			try {
				final int w = bounds.width;
				final int h = bounds.height;
				VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

				buffer = context.createVideoBuffer(bounds);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, buffer.getTexture(), 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, input.getTexture());

				gl.glColor4f(1, 1, 1, 1);

				program.useProgram(new Runnable() {
					public void run() {
						gl.glUniform1i(program.getUniformLocation("texture"), 0);
						gl.glUniform1f(program.getUniformLocation("brightness"), (float)(brightness/255));
						gl.glUniform1f(program.getUniformLocation("contrastSlope"), (float)contrastSlope);

						gl.glBegin(GL2.GL_QUADS);
						gl.glTexCoord2f(0, 0);
						gl.glVertex2f(0, 0);
						gl.glTexCoord2f(w, 0);
						gl.glVertex2f(w, 0);
						gl.glTexCoord2f(w, h);
						gl.glVertex2f(w, h);
						gl.glTexCoord2f(0, h);
						gl.glVertex2f(0, h);
						gl.glEnd();
					}
				});

				IVideoBuffer output = buffer;
				buffer = null;
				return output;

			} finally {
				gl.glPopAttrib();

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
			}

		} finally {
			if (input != null) input.dispose();
			if (buffer != null) buffer.dispose();
		}
	}

}
