/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.BlendMode;
import ch.kuramo.javie.effects.BlendModeShaders;
import ch.kuramo.javie.effects.VideoEffectUtil;

public abstract class AudioDrawing {

	public enum Channel { MONO, LEFT, RIGHT }

	public enum Style { BARS, LINES, DOTS }


	private static volatile double maxSmoothThickness;

	private static double getMaxSmoothThickness(GL2 gl) {
		double max = maxSmoothThickness;
		if (max == 0) {
			synchronized (AudioDrawing.class) {
				max = maxSmoothThickness;
				if (max == 0) {
					float[] range = new float[2];
					gl.glGetFloatv(GL2.GL_SMOOTH_LINE_WIDTH_RANGE, range, 0);
					max = range[1];
					gl.glGetFloatv(GL2.GL_SMOOTH_POINT_SIZE_RANGE, range, 0);
					max = Math.min(max, range[1]);
					maxSmoothThickness = max;
				}
			}
		}
		return max;
	}


	protected final IVideoEffectContext context;

	protected final BlendModeShaders blendModeShaders;


	protected AudioDrawing(IVideoEffectContext context, IShaderRegistry shaders) {
		this.context = context;
		blendModeShaders = BlendModeShaders.forSrcUnmult(context, shaders);
	}

	protected IVideoBuffer doDrawing(BlendMode blendMode, double opacity) {
		IVideoBuffer original = null;
		IVideoBuffer drawing = null;

		try {
			VideoBounds bounds;

			if (blendMode == BlendMode.NONE) {
				bounds = context.getPreviousBounds();
			} else {
				original = context.doPreviousEffect();
				bounds = original.getBounds();
			}

			if (bounds.isEmpty()) {
				IVideoBuffer result;
				if (original != null) {
					result = original;
					original = null;
				} else {
					result = context.createVideoBuffer(bounds);
				}
				return result;
			}

			GL2 gl = context.getGL().getGL2();
			GLU glu = context.getGLU();

			drawing = context.createVideoBuffer(bounds);
			VideoEffectUtil.clearTexture(drawing, gl);

			draw(drawing, gl, glu);

			return blendModeShaders.blend(drawing, original, blendMode, opacity);

		} finally {
			if (original != null) original.dispose();
			if (drawing != null) drawing.dispose();
		}
	}

	protected abstract void draw(IVideoBuffer drawing, GL2 gl, GLU glu);


	protected interface DataProvider {
		int getDataCount();
		float getDataLower(int index);
		float getDataUpper(int index);
		float getData(int index);
	}

	protected void draw(
			DataProvider dataProvider, IVideoBuffer dst, Vec2d startPoint, Vec2d endPoint,
			double height, double thickness, Color color, Style style, boolean smoothing, GL2 gl, GLU glu) {

		if (smoothing) {
			thickness = Math.min(thickness, getMaxSmoothThickness(gl));
		}
		thickness = Math.max(1e-10, thickness);

		double dx = endPoint.x-startPoint.x;
		double dy = endPoint.y-startPoint.y;


		VideoBounds bounds = dst.getBounds();
		VideoEffectUtil.ortho2D(gl, glu, bounds.width, bounds.height);
		gl.glTranslatef((float)-bounds.x, (float)-bounds.y, 0);

		gl.glTranslatef((float)startPoint.x, (float)startPoint.y, 0);
		gl.glRotatef((float)Math.toDegrees(Math.atan2(dy, dx)), 0, 0, 1);
		gl.glScalef((float)Math.sqrt(dx*dx+dy*dy), (float)height, 1);

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, dst.getTexture(), 0);
		gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

		gl.glPushAttrib(
				  GL2.GL_HINT_BIT			// GL_LINE_SMOOTH_HINT, GL_POINT_SMOOTH_HINT
				| GL2.GL_CURRENT_BIT		// GL_CURRENT_COLOR
				| GL2.GL_COLOR_BUFFER_BIT	// GL_BLEND, GL_BLEND_SRC_RGB, GL_BLEND_SRC_ALPHA, GL_BLEND_DST_RGB, GL_BLEND_DST_ALPHA, GL_BLEND_EQUATION
				| GL2.GL_POINT_BIT			// GL_POINT_SIZE, GL_POINT_SMOOTH
				| GL2.GL_LINE_BIT			// GL_LINE_WIDTH, GL_LINE_SMOOTH
				| GL2.GL_DEPTH_BUFFER_BIT	// GL_DEPTH_TEST
				| GL2.GL_ENABLE_BIT);		// GL_DEPTH_TEST, GL_BLEND, GL_LINE_SMOOTH, GL_POINT_SMOOTH

		try {
			if (smoothing) {
				gl.glDisable(GL2.GL_DEPTH_TEST);	// もともと無効になっているはずだけど。

				gl.glEnable(GL2.GL_LINE_SMOOTH);
				gl.glEnable(GL2.GL_POINT_SMOOTH);
				gl.glHint(GL2.GL_LINE_SMOOTH_HINT, GL2.GL_NICEST);
				gl.glHint(GL2.GL_POINT_SMOOTH_HINT, GL2.GL_NICEST);

				gl.glEnable(GL2.GL_BLEND);
				gl.glBlendFunc(GL2.GL_ONE, GL2.GL_ONE);
				gl.glBlendEquation(GL2.GL_MAX);
			}

			gl.glLineWidth((float)thickness);
			gl.glPointSize((float)thickness);

			float a = (float)color.a;
			float r = (float)color.r * a;
			float g = (float)color.g * a;
			float b = (float)color.b * a;
			gl.glColor4f(r, g, b, a);

			int dataCount = dataProvider.getDataCount();

			switch (style) {
				case BARS: {
					float adjustment = (float)(smoothing ? 0 : 1/height);

					gl.glBegin(GL2.GL_LINES);
					for (int i = 0; i < dataCount; ++i) {
						gl.glVertex2f((float)i/(dataCount-1), dataProvider.getDataLower(i));
						gl.glVertex2f((float)i/(dataCount-1), dataProvider.getDataUpper(i) - adjustment);
					}
					gl.glEnd();

					if (smoothing) {
						gl.glBegin(GL2.GL_POINTS);
						for (int i = 0; i < dataCount; ++i) {
							gl.glVertex2f((float)i/(dataCount-1), dataProvider.getDataLower(i));
							gl.glVertex2f((float)i/(dataCount-1), dataProvider.getDataUpper(i) - adjustment);
						}
						gl.glEnd();
					}
					break;
				}

				case LINES:
					gl.glBegin(GL2.GL_LINE_STRIP);
					if (dataCount > 1) {
						for (int i = 0; i < dataCount; ++i) {
							gl.glVertex2f((float)i/(dataCount-1), dataProvider.getData(i));
						}
					} else {
						float data = dataProvider.getData(0);
						gl.glVertex2f(0, data);
						gl.glVertex2f(1, data);
					}
					gl.glEnd();

					if (!smoothing) break;
					// fall through

				case DOTS:
					gl.glBegin(GL2.GL_POINTS);
					if (dataCount > 1) {
						for (int i = 0; i < dataCount; ++i) {
							gl.glVertex2f((float)i/(dataCount-1), dataProvider.getData(i));
						}
					} else if (style == Style.LINES) {
						float data = dataProvider.getData(0);
						gl.glVertex2f(0, data);
						gl.glVertex2f(1, data);
					} else {
						gl.glVertex2f(0.5f, dataProvider.getData(0));
					}
					gl.glEnd();
					break;
			}

		} finally {
			gl.glPopAttrib();

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
		}
	}

}
