/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.BlendMode;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.AudioWaveform", category=Effect.GENERATE)
public class AudioWaveform extends AudioDrawing {

	@Property
	private IAnimatableLayerReference audioLayer;

	@Property
	private IAnimatableVec2d startPoint;

	@Property
	private IAnimatableVec2d endPoint;

	@Property(value="120", min="1", max="96000")
	private IAnimatableInteger displayedSamples;

	@Property(value="120", min="1")
	private IAnimatableDouble height;

	@Property(value="125", min="0", max="30000")
	private IAnimatableDouble audioDuration;

	@Property(value="0", min="-30000", max="30000")
	private IAnimatableDouble audioOffset;

	@Property(value="3", min="0", max="100")
	private IAnimatableDouble thickness;

	@Property("0,1,0")
	private IAnimatableColor color;

	@Property("MONO")
	private IAnimatableEnum<Channel> channel;

	@Property("LINES")
	private IAnimatableEnum<Style> style;

	@Property("true")
	private IAnimatableBoolean smoothing;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableEnum<BlendMode> blendMode;


	private final IArrayPools arrayPools;


	@Inject
	public AudioWaveform(IVideoEffectContext context, IShaderRegistry shaders, IArrayPools arrayPools) {
		super(context, shaders);
		this.arrayPools = arrayPools;
	}

	public IVideoBuffer doVideoEffect() {
		return doDrawing(context.value(blendMode), context.value(opacity)/100);
	}

	@Override
	protected void draw(IVideoBuffer drawing, GL2 gl, GLU glu) {
		IAudioBuffer audio = null;
		IArray<float[]> samples = null;

		try {
			Resolution resolution = context.getVideoResolution();

			Vec2d startPoint = resolution.scale(context.value(this.startPoint));
			Vec2d endPoint = resolution.scale(context.value(this.endPoint));
			final int displayedSamples = context.value(this.displayedSamples);
			double height = resolution.scale(context.value(this.height));
			double audioDuration = context.value(this.audioDuration) / 1000;
			double audioOffset = context.value(this.audioOffset) / 1000;
			double thickness = resolution.scale(context.value(this.thickness));
			Color color = context.value(this.color);
			Channel channel = context.value(this.channel);
			Style style = context.value(this.style);
			boolean smoothing = context.value(this.smoothing);

			AudioMode audioMode = context.getAudioMode();
			int audioFrameCount = (int)(audioDuration*audioMode.sampleRate);
			double sampleInterval = (double)audioFrameCount / displayedSamples;

			Time startOffset = new Time((long)((audioOffset-audioDuration/2)*audioMode.sampleRate), audioMode.sampleRate);
			Time startTime = startOffset.add(context.getTime());
			int startIndex = (int)Math.floor(startTime.timeValue/sampleInterval);
			startTime = new Time(Math.round(startIndex*sampleInterval), startTime.timeScale);

			context.setTime(startTime);
			context.setAudioFrameCount(audioFrameCount);

			audio = context.getLayerAudioChunk(audioLayer);
			samples = arrayPools.getFloatArray(displayedSamples*2);

			if (audio == null) {
				samples.clear();
			} else {
				sample(audio, samples, channel);
			}

			final float[] sampleArray = samples.getArray();
			final int evenOdd = Math.abs(startIndex) % 2;

			DataProvider dp = new DataProvider() {
				public int getDataCount()				{ return displayedSamples; }
				public float getDataUpper(int index)	{ return sampleArray[index*2]; }
				public float getDataLower(int index)	{ return sampleArray[index*2+1]; }
				public float getData(int index)			{ return sampleArray[index*2+(evenOdd+index)%2]; }
			};

			draw(dp, drawing, startPoint, endPoint,
					height, thickness, color, style, smoothing, gl, glu);

		} finally {
			if (audio != null) audio.dispose();
			if (samples != null) samples.release();
		}
	}

	private void sample(IAudioBuffer audio, IArray<float[]> samples, Channel channel) {
		Object audioData = audio.getData();
		int audioFrames = audio.getFrameCount();

		float[] sampleArray = samples.getArray();
		int numSamples = samples.getLength() / 2;

		switch (audio.getAudioMode().dataType) {
			case SHORT: {
				short[] audioArray = (short[]) audioData;
				float left = (channel == Channel.RIGHT) ? 0 : 1f/Short.MAX_VALUE;
				float right = (channel == Channel.LEFT) ? 0 : 1f/Short.MAX_VALUE;

				for (int i = 0; i < numSamples; ++i) {
					int p = i*2, q = p+1;
					sampleArray[p] = Float.POSITIVE_INFINITY;
					sampleArray[q] = Float.NEGATIVE_INFINITY;
					for (int j = i*audioFrames/numSamples*2, k = (i+1)*audioFrames/numSamples*2; j < k; j+=2) {
						float sample = left*audioArray[j] + right*audioArray[j+1];
						sampleArray[p] = Math.min(sampleArray[p], sample);
						sampleArray[q] = Math.max(sampleArray[q], sample);
					}
				}
				break;
			}

			case INT: {
				int[] audioArray = (int[]) audioData;
				float left = (channel == Channel.RIGHT) ? 0 : 1f/Integer.MAX_VALUE;
				float right = (channel == Channel.LEFT) ? 0 : 1f/Integer.MAX_VALUE;

				for (int i = 0; i < numSamples; ++i) {
					int p = i*2, q = p+1;
					sampleArray[p] = Float.POSITIVE_INFINITY;
					sampleArray[q] = Float.NEGATIVE_INFINITY;
					for (int j = i*audioFrames/numSamples*2, k = (i+1)*audioFrames/numSamples*2; j < k; j+=2) {
						float sample = left*audioArray[j] + right*audioArray[j+1];
						sampleArray[p] = Math.min(sampleArray[p], sample);
						sampleArray[q] = Math.max(sampleArray[q], sample);
					}
				}
				break;
			}

			case FLOAT: {
				float[] audioArray = (float[]) audioData;
				float left = (channel == Channel.RIGHT) ? 0 : 1;
				float right = (channel == Channel.LEFT) ? 0 : 1;

				for (int i = 0; i < numSamples; ++i) {
					int p = i*2, q = p+1;
					sampleArray[p] = Float.POSITIVE_INFINITY;
					sampleArray[q] = Float.NEGATIVE_INFINITY;
					for (int j = i*audioFrames/numSamples*2, k = (i+1)*audioFrames/numSamples*2; j < k; j+=2) {
						float sample = left*audioArray[j] + right*audioArray[j+1];
						sampleArray[p] = Math.min(sampleArray[p], sample);
						sampleArray[q] = Math.max(sampleArray[q], sample);
					}
				}
				break;
			}

			default:
				throw new UnsupportedOperationException(
						"unsupported AudioMode.DataType: " + audio.getAudioMode().dataType);
		}
	}

}
