/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import java.nio.FloatBuffer;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;
import javax.media.opengl.glu.GLUquadric;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.BlendMode;
import ch.kuramo.javie.effects.BlendModeShaders;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Circle", category=Effect.GENERATE)
public class Circle {

	public enum Edge {
		NONE,
		EDGE_RADIUS,
		THICKNESS,
		P_THICKNESS,				// PはProportionalのP
		P_THICKNESS_P_FEATHER
	}


	@Property
	private IAnimatableVec2d location;

	@Property(value="75", min="0")
	private IAnimatableDouble radius;

	@Property("0.5")
	private IAnimatableVec2d anchor;

	@Property
	private IAnimatableEnum<Edge> edge;

	@Property(value="10", min="0")
	private IAnimatableDouble edgeRadius;

	@Property(value="10", min="0", max="4000")
	private IAnimatableDouble thickness;

	@Property(min="0", max="4000")
	private IAnimatableDouble outerFeather;

	@Property(min="0", max="4000")
	private IAnimatableDouble innerFeather;

	@Property
	private IAnimatableBoolean invert;

	@Property("1,1,1")
	private IAnimatableColor color;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableEnum<BlendMode> blendMode;


	private final IVideoEffectContext context;

	private final IArrayPools arrayPools;

	private final IShaderProgram circleSamplerProgram;

	private final IShaderProgram invertSamplerProgram;

	private final BlendModeShaders blendModeShaders;

	@Inject
	public Circle(IVideoEffectContext context, IArrayPools arrayPools, IShaderRegistry shaders) {
		this.context = context;
		this.arrayPools = arrayPools;

		circleSamplerProgram = shaders.getProgram(Circle.class, "CIRCLE_SAMPLER");
		invertSamplerProgram = shaders.getProgram(Circle.class, "INVERT_SAMPLER");
		blendModeShaders = BlendModeShaders.forPremult(context, shaders);
	}

	public IVideoBuffer doVideoEffect() {
		final GL2 gl = context.getGL().getGL2();
		final GLU glu = context.getGLU();

		IVideoBuffer original = null;
		IVideoBuffer circle = null;
		int[] tex1d = null;
		GLUquadric quadric = null;
		try {
			final VideoBounds bounds;

			BlendMode blendMode = context.value(this.blendMode);
			if (blendMode == BlendMode.NONE) {
				bounds = context.getPreviousBounds();
			} else {
				original = context.doPreviousEffect();
				bounds = original.getBounds();
			}

			if (bounds.isEmpty()) {
				if (original == null) {
					original = context.createVideoBuffer(bounds);
					//VideoEffectUtil.clearTexture(original, gl);
					// TODO 空のVideoBufferではあるが、実際には1x1のテクスチャが作成されるのでそれをクリアする必要がある？
					// TODO 他にも同様に空のVideoBufferを作成していてクリアしていない箇所がある。
				}
				IVideoBuffer result = original;
				original = null;
				return result;
			}

			Resolution resolution = context.getVideoResolution();

			Vec2d location = context.value(this.location);
			double radius = context.value(this.radius);
			Vec2d anchor = context.value(this.anchor);

			final Vec2d center = resolution.scale(new Vec2d(location.x-radius*(anchor.x-0.5)*2, location.y-radius*(anchor.y-0.5)*2));
			radius = resolution.scale(radius);

			Edge edge = context.value(this.edge);
			boolean invert = context.value(this.invert);
			Color color = context.value(this.color);
			double opacity = context.value(this.opacity) / 100;

			double edgeRadius = 0;
			double thickness = 0;
			double innerFeather = 0;
			double outerFeather = 0;

			switch (edge) {
				case NONE:
					outerFeather = resolution.scale(context.value(this.outerFeather));
					break;

				case EDGE_RADIUS:
					edgeRadius = resolution.scale(context.value(this.edgeRadius));
					if (edgeRadius > radius) {
						double tmp = edgeRadius;
						edgeRadius = radius;
						radius = tmp;
					}
					innerFeather = resolution.scale(context.value(this.innerFeather));
					outerFeather = resolution.scale(context.value(this.outerFeather));
					break;

				case THICKNESS:
					thickness = resolution.scale(context.value(this.thickness));
					innerFeather = resolution.scale(context.value(this.innerFeather));
					outerFeather = resolution.scale(context.value(this.outerFeather));
					break;

				case P_THICKNESS:
					// radiusがすでにスケール済みなので、thicknessはスケールしない。
					thickness = context.value(this.thickness) * radius / 100;
					innerFeather = resolution.scale(context.value(this.innerFeather));
					outerFeather = resolution.scale(context.value(this.outerFeather));
					break;

				case P_THICKNESS_P_FEATHER:
					// radiusがすでにスケール済みなのでスケールしない。
					thickness = context.value(this.thickness) * radius / 100;
					innerFeather = context.value(this.innerFeather) * radius / 100;
					outerFeather = context.value(this.outerFeather) * radius / 100;
					break;
			}

			final double innerRadius;
			final double outerRadius;
			final double innerMiddle;
			final double outerMiddle;
			final int[] texSize = new int[2];

			switch (edge) {
				case NONE:
					innerRadius = 0;
					outerRadius = radius + outerFeather;
					innerMiddle = 0;
					outerMiddle = radius;
					tex1d = createCircleTextures(texSize, 0, 0, outerFeather, gl);
					break;

				case EDGE_RADIUS:
					innerRadius = Math.max(0, edgeRadius - innerFeather);
					outerRadius = radius + outerFeather;
					innerMiddle = edgeRadius;
					outerMiddle = radius;
					tex1d = createCircleTextures(texSize, innerMiddle, innerFeather, outerFeather, gl);
					break;

				default:
					innerRadius = Math.max(0, radius - thickness*0.5 - innerFeather);
					outerRadius = radius + thickness*0.5 + outerFeather;
					innerMiddle = radius - thickness*0.5;
					outerMiddle = radius + thickness*0.5;
					tex1d = createCircleTextures(texSize, innerMiddle, innerFeather, outerFeather, gl);
					break;
			}

			gl.glPushAttrib(GL2.GL_CURRENT_BIT | GL2.GL_TEXTURE_BIT);
			try {
				circle = context.createVideoBuffer(bounds);
				VideoEffectUtil.clearTexture(circle, gl);

				int w = bounds.width;
				int h = bounds.height;
				VideoEffectUtil.ortho2D(gl, glu, w, h);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, circle.getTexture(), 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				final float a = (float)(color.a * (blendMode == BlendMode.NONE ? opacity : 1));
				final float r = (float)color.r * a;
				final float g = (float)color.g * a;
				final float b = (float)color.b * a;
				gl.glColor4f(r, g, b, a);

				if (invert) {
					gl.glBegin(GL2.GL_QUADS);
					gl.glVertex2f(0, 0);
					gl.glVertex2f(w, 0);
					gl.glVertex2f(w, h);
					gl.glVertex2f(0, h);
					gl.glEnd();
				}

				gl.glTranslatef((float)(center.x-bounds.x), (float)(center.y-bounds.y), 0);

				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_1D, tex1d[0]);
				gl.glActiveTexture(GL2.GL_TEXTURE1);
				gl.glBindTexture(GL2.GL_TEXTURE_1D, tex1d[1]);

				quadric = glu.gluNewQuadric(/* TODO これのオーバーロードされたメソッド引数 useGLSL って何だろう？ */);
				glu.gluQuadricDrawStyle(quadric, GLU.GLU_FILL);
				glu.gluQuadricNormals(quadric, GLU.GLU_NONE);

				final GLUquadric q = quadric;
				final IShaderProgram program = invert ? invertSamplerProgram : circleSamplerProgram;
				program.useProgram(new Runnable() {
					public void run() {
						gl.glUniform1i(program.getUniformLocation("texInner"), 0);
						gl.glUniform1i(program.getUniformLocation("texOuter"), 1);
						gl.glUniform2f(program.getUniformLocation("texSize"), texSize[0], texSize[1]);
						gl.glUniform2f(program.getUniformLocation("texOffset"),
								(float)(texSize[0]*0.5-innerMiddle), (float)(texSize[1]*0.5-outerMiddle));
						gl.glUniform2f(program.getUniformLocation("center"),
								(float)(center.x-bounds.x), (float)(center.y-bounds.y));
						gl.glUniform4f(program.getUniformLocation("color"), r, g, b, a);

						// TODO slicesはこれで妥当か？ ちょっと多すぎるかも。
						int slices = (int)Math.ceil(2*Math.PI*outerRadius);
						glu.gluDisk(q, Math.max(0, innerRadius-2), outerRadius+2, slices, 1);
					}
				});

				IVideoBuffer result;
				if (blendMode == BlendMode.NONE) {
					result = circle;
					circle = null;
				} else {
					result = blendModeShaders.blend(circle, original, blendMode, opacity);
				}
				return result;

			} finally {
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				gl.glPopAttrib();
			}

		} finally {
			if (original != null) original.dispose();
			if (circle != null) circle.dispose();
			if (tex1d != null) gl.glDeleteTextures(2, tex1d, 0);
			if (quadric != null) glu.gluDeleteQuadric(quadric);
		}
	}

	private int[] createCircleTextures(int[] texSize,
			double innerMiddle, double innerFeather, double outerFeather, GL2 gl) {

		int[] tex1d = null;
		try {
			tex1d = new int[2];
			gl.glGenTextures(2, tex1d, 0);

			if (innerMiddle <= 0 && innerFeather == 0) {
				copyTextureData(new float[] { 1, 1 }, 2, tex1d[0], gl);
				texSize[0] = 2;
			} else {
				texSize[0] = createCircleTexture(true, innerFeather, tex1d[0], gl);
			}

			texSize[1] = createCircleTexture(false, outerFeather, tex1d[1], gl);

			int[] result = tex1d;
			tex1d = null;
			return result;

		} finally {
			if (tex1d != null) gl.glDeleteTextures(2, tex1d, 0);
		}
	}

	private int createCircleTexture(boolean inner, double feather, int tex1d, GL2 gl) {
		IArray<float[]> data = null;
		try {
			int halfLen = (int)Math.ceil(feather) + 1;

			data = arrayPools.getFloatArray(halfLen*2);
			float[] array = data.getArray();
			int arrayLen = data.getLength();

			double lowerOutside = inner ? 0 : 1;
			double upperOutside = inner ? 1 : 0;
			double offset = halfLen + (inner ? -feather : feather);

			for (int i = 0; i < arrayLen; ++i) {
				double r = i + 0.5;
				double d = (r <= halfLen-feather) ? lowerOutside
						 : (r >= halfLen+feather) ? upperOutside
						 : 0.5*(1-Math.cos(2*Math.PI*(r-offset)/(4*feather)));

				array[i] = (float) d;
			}

			copyTextureData(array, arrayLen, tex1d, gl);
			return arrayLen;

		} finally {
			if (data != null) data.release();
		}
	}

	private void copyTextureData(float[] array, int arrayLen, int tex1d, GL2 gl) {
		try {
			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, tex1d);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_EDGE);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_EDGE);

			FloatBuffer buffer = FloatBuffer.wrap(array, 0, arrayLen);

			switch (context.getColorMode()) {
				case RGBA8:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA8, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA16:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA16, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA16_FLOAT:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA16F, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA32_FLOAT:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA32F, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				default:
					throw new RuntimeException("unknown ColorMode: " + context.getColorMode());
			}
		} finally {
			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, 0);
		}
	}

	private static final String[] createSamplerProgram(boolean invert) {
		return new String[] {
				"uniform sampler1D texInner;",
				"uniform sampler1D texOuter;",
				"uniform vec2 texSize;",
				"uniform vec2 texOffset;",
				"uniform vec2 center;",
				"uniform vec4 color;",		// ATIでgl_Colorを使うには頂点シェーダで明示的にglFrontColorを設定する必要がある。
				"",							// ここではそれをせず、uniform変数で色を渡している。
				"void main(void)",
				"{",
				"	float d = distance(center, gl_FragCoord.st);",
				"	vec2 texCoord = (vec2(d, d) + texOffset) / texSize;",
				"",
				"	float inner = texture1D(texInner, texCoord.x).a;",
				"	float outer = texture1D(texOuter, texCoord.y).a;",
				"",
	   invert ? "	gl_FragColor = color * (1.0-inner*outer);"
			  : "	gl_FragColor = color * (inner*outer);",
				"}"
		};
	}

	@ShaderSource
	public static final String[] CIRCLE_SAMPLER = createSamplerProgram(false);

	@ShaderSource
	public static final String[] INVERT_SAMPLER = createSamplerProgram(true);

}
