/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import java.nio.FloatBuffer;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.BlendMode;
import ch.kuramo.javie.effects.BlendModeShaders;

public abstract class GradientBase {

	public enum ColorSpace {
		RGB,
		HSL,
		HSL_REVERSE_HUE
	}

	public enum RepeatMode {
		NONE,
		CLAMP_TO_EDGE,
		SAWTOOTH,
		TRIANGLE
	}


	protected final IVideoEffectContext context;

	protected final IArrayPools arrayPools;

	protected final BlendModeShaders blendModeShaders;


	protected GradientBase(IVideoEffectContext context, IArrayPools arrayPools, IShaderRegistry shaders) {
		this.context = context;
		this.arrayPools = arrayPools;
		blendModeShaders = BlendModeShaders.forPremult(context, shaders);
	}

	protected IVideoBuffer doGradientEffect(BlendMode blendMode, double opacity) {
		IVideoBuffer original = null;
		IVideoBuffer gradient = null;
		try {
			VideoBounds bounds;

			if (blendMode == BlendMode.NONE) {
				bounds = context.getPreviousBounds();
			} else {
				original = context.doPreviousEffect();
				bounds = original.getBounds();
			}

			if (bounds.isEmpty()) {
				if (original == null) {
					original = context.createVideoBuffer(bounds);
					//VideoEffectUtil.clearTexture(original, gl);
					// TODO 空のVideoBufferではあるが、実際には1x1のテクスチャが作成されるのでそれをクリアする必要がある？
					// TODO 他にも同様に空のVideoBufferを作成していてクリアしていない箇所がある。
				}
				IVideoBuffer result = original;
				original = null;
				return result;
			}

			gradient = createGradient(bounds);

			IVideoBuffer result;
			if (blendMode == BlendMode.NONE && opacity == 1) {
				result = gradient;
				gradient = null;
			} else {
				result = blendModeShaders.blend(gradient, original, blendMode, opacity);
			}
			return result;

		} finally {
			if (original != null) original.dispose();
			if (gradient != null) gradient.dispose();
		}
	}

	protected abstract IVideoBuffer createGradient(VideoBounds bounds);


	protected int[] createRGBGradientTexture(Color colorA, Color colorB, int texSize, RepeatMode repeatMode, GL2 gl) {
		if (texSize <= 0) {
			throw new IllegalArgumentException("texSize must be greater than 0");
		}

		IArray<float[]> data = null;
		int[] tex1d = null;
		try {
			data = arrayPools.getFloatArray(texSize*4);
			float[] array = data.getArray();

			if (texSize == 1) {
				double a = colorA.a*0.5 + colorB.a*0.5;
				array[0] = (float)((colorA.r*0.5 + colorB.r*0.5)*a);
				array[1] = (float)((colorA.g*0.5 + colorB.g*0.5)*a);
				array[2] = (float)((colorA.b*0.5 + colorB.b*0.5)*a);
				array[3] = (float)a;

			} else {
				for (int i = 0; i < texSize; ++i) {
					double t = i / (texSize-1.0);
					double a = colorA.a*(1-t) + colorB.a*t;
					array[i*4  ] = (float)((colorA.r*(1-t) + colorB.r*t)*a);
					array[i*4+1] = (float)((colorA.g*(1-t) + colorB.g*t)*a);
					array[i*4+2] = (float)((colorA.b*(1-t) + colorB.b*t)*a);
					array[i*4+3] = (float)a;
				}
			}

			tex1d = new int[1];
			gl.glGenTextures(1, tex1d, 0);
			copyTextureData(array, data.getLength(), repeatMode, tex1d[0], gl);

			int[] result = tex1d;
			tex1d = null;
			return result;

		} finally {
			if (data != null) data.release();
			if (tex1d != null) gl.glDeleteTextures(1, tex1d, 0);
		}
	}

	protected int[] createHSLGradientTexture(Color colorA, Color colorB, boolean reverseHue, int texSize, RepeatMode repeatMode, GL2 gl) {
		if (texSize <= 0) {
			throw new IllegalArgumentException("texSize must be greater than 0");
		}

		IArray<float[]> data = null;
		int[] tex1d = null;
		try {
			data = arrayPools.getFloatArray(texSize*4);
			float[] array = data.getArray();

			double[] hslA = new double[3];
			double[] hslB = new double[3];
			rgb2hsl(colorA, hslA);
			rgb2hsl(colorB, hslB);

			if (reverseHue) {
				if (hslA[0] <= hslB[0]) {
					hslA[0] += 1;
				}
			} else {
				if (hslA[0] > hslB[0]) {
					hslB[0] += 1;
				}
			}

			double[] hsl = new double[3];
			double[] rgb = new double[3];

			if (texSize == 1) {
				hsl[0] = hslA[0]*0.5 + hslB[0]*0.5;
				hsl[1] = hslA[1]*0.5 + hslB[1]*0.5;
				hsl[2] = hslA[2]*0.5 + hslB[2]*0.5;
				hsl2rgb(hsl, rgb);
				double a = colorA.a*0.5 + colorB.a*0.5;
				array[0] = (float)(rgb[0]*a);
				array[1] = (float)(rgb[1]*a);
				array[2] = (float)(rgb[2]*a);
				array[3] = (float)a;
			} else {
				for (int i = 0; i < texSize; ++i) {
					double t = i / (texSize-1.0);
					hsl[0] = hslA[0]*(1-t) + hslB[0]*t;
					hsl[1] = hslA[1]*(1-t) + hslB[1]*t;
					hsl[2] = hslA[2]*(1-t) + hslB[2]*t;
					hsl2rgb(hsl, rgb);
					double a = colorA.a*(1-t) + colorB.a*t;
					array[i*4  ] = (float)(rgb[0]*a);
					array[i*4+1] = (float)(rgb[1]*a);
					array[i*4+2] = (float)(rgb[2]*a);
					array[i*4+3] = (float)a;
				}
			}

			tex1d = new int[1];
			gl.glGenTextures(1, tex1d, 0);
			copyTextureData(array, data.getLength(), repeatMode, tex1d[0], gl);

			int[] result = tex1d;
			tex1d = null;
			return result;

		} finally {
			if (data != null) data.release();
			if (tex1d != null) gl.glDeleteTextures(1, tex1d, 0);
		}
	}

	private void rgb2hsl(Color rgb, double[] hsl) {
		double min = Math.min(Math.min(rgb.r, rgb.g), rgb.b);
		double max = Math.max(Math.max(rgb.r, rgb.g), rgb.b);
		double dmax = max - min;

		double luma = (max + min)*0.5;
		double hue, sat;

		if (dmax == 0) {
			hue = sat = 0;
		} else {
			sat = (luma < 0.5) ? dmax/(max+min) : dmax/(2-max-min);

			double dr = ((max-rgb.r)/6 + dmax/2)/dmax;
			double dg = ((max-rgb.g)/6 + dmax/2)/dmax;
			double db = ((max-rgb.b)/6 + dmax/2)/dmax;

			hue = (rgb.r == max) ? db-dg
				: (rgb.g == max) ? 1./3 + dr-db
				:				   2./3 + dg-dr;

			if (hue < 0) hue += 1;
			else if (hue > 1) hue -= 1;
		}

		hsl[0] = hue;
		hsl[1] = sat;
		hsl[2] = luma;
	}

	private void hsl2rgb(double[] hsl, double[] rgb) {
		double hue = hsl[0];
		double sat = hsl[1];
		double luma = hsl[2];

		if (hue > 1) hue -= 1;

		if (sat == 0) {
			rgb[0] = rgb[1] = rgb[2] = luma;
		} else {
			double t2 = (luma < 0.5) ? luma*(1+sat) : luma+sat-luma*sat;
			double t1 = luma*2 - t2;

			rgb[0] = hue2rgb(t1, t2, hue+1./3);
			rgb[1] = hue2rgb(t1, t2, hue);
			rgb[2] = hue2rgb(t1, t2, hue-1./3);
		}
	}

	private double hue2rgb(double t1, double t2, double hue) {
		if (hue < 0) hue += 1;
		else if (hue > 1) hue -= 1;

		return (hue*6 < 1) ? t1 + (t2-t1)*6*hue
			 : (hue*2 < 1) ? t2
			 : (hue*3 < 2) ? t1 + (t2-t1)*(2./3-hue)*6
			 :				 t1;
	}


	private static final float[] FLOAT0000 = new float[] { 0, 0, 0, 0 };

	protected void copyTextureData(float[] array, int arrayLen, RepeatMode repeatMode, int tex1d, GL2 gl) {
		try {
			int wrapMode;
			switch (repeatMode) {
				case CLAMP_TO_EDGE:	wrapMode = GL2.GL_CLAMP_TO_EDGE;	break;
				case SAWTOOTH:		wrapMode = GL2.GL_REPEAT;			break;
				case TRIANGLE:		wrapMode = GL2.GL_MIRRORED_REPEAT;	break;
				default:			wrapMode = GL2.GL_CLAMP_TO_BORDER;	break;
			}

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, tex1d);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_S, wrapMode);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_EDGE);

			if (wrapMode == GL2.GL_CLAMP_TO_BORDER) {
				gl.glTexParameterfv(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_BORDER_COLOR, FLOAT0000, 0);
			}

			int width = arrayLen / 4;
			FloatBuffer buffer = FloatBuffer.wrap(array, 0, arrayLen);

			switch (context.getColorMode()) {
				case RGBA8:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_RGBA8, width, 0, GL2.GL_RGBA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA16:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_RGBA16, width, 0, GL2.GL_RGBA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA16_FLOAT:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_RGBA16F, width, 0, GL2.GL_RGBA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA32_FLOAT:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_RGBA32F, width, 0, GL2.GL_RGBA, GL2.GL_FLOAT, buffer);
					break;

				default:
					throw new RuntimeException("unknown ColorMode: " + context.getColorMode());
			}

		} finally {
			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, 0);
		}
	}

}
