/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import java.nio.FloatBuffer;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.BlendMode;
import ch.kuramo.javie.effects.BlendModeShaders;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Grid", category=Effect.GENERATE)
public class Grid {

	@Property
	private IAnimatableVec2d anchor;

	@Property(value="100", min="0", max="4000")
	private IAnimatableVec2d gridSize;

	@Property(value="5", min="0", max="4000")
	private IAnimatableVec2d border;

	@Property(value="0", min="0", max="400")
	private IAnimatableVec2d feather;

	@Property
	private IAnimatableBoolean invert;

	@Property("1,1,1")
	private IAnimatableColor color;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableEnum<BlendMode> blendMode;


	private final IVideoEffectContext context;

	private final IArrayPools arrayPools;

	private final IShaderProgram gridSamplerProgram;

	private final IShaderProgram invertSamplerProgram;

	private final BlendModeShaders blendModeShaders;

	@Inject
	public Grid(IVideoEffectContext context, IArrayPools arrayPools, IShaderRegistry shaders) {
		this.context = context;
		this.arrayPools = arrayPools;

		gridSamplerProgram = shaders.getProgram(Grid.class, "GRID_SAMPLER");
		invertSamplerProgram = shaders.getProgram(Grid.class, "INVERT_SAMPLER");
		blendModeShaders = BlendModeShaders.forPremult(context, shaders);
	}

	public IVideoBuffer doVideoEffect() {
		final GL2 gl = context.getGL().getGL2();

		IVideoBuffer original = null;
		IVideoBuffer grid = null;
		int[] tex1d = null;
		try {
			final VideoBounds bounds;

			BlendMode blendMode = context.value(this.blendMode);
			if (blendMode == BlendMode.NONE) {
				bounds = context.getPreviousBounds();
			} else {
				original = context.doPreviousEffect();
				bounds = original.getBounds();
			}

			if (bounds.isEmpty()) {
				if (original == null) {
					original = context.createVideoBuffer(bounds);
					//VideoEffectUtil.clearTexture(original, gl);
					// TODO 空のVideoBufferではあるが、実際には1x1のテクスチャが作成されるのでそれをクリアする必要がある？
					// TODO 他にも同様に空のVideoBufferを作成していてクリアしていない箇所がある。
				}
				IVideoBuffer result = original;
				original = null;
				return result;
			}

			Resolution resolution = context.getVideoResolution();

			final Vec2d anchor = resolution.scale(context.value(this.anchor));
			final Vec2d gridSize = resolution.scale(context.value(this.gridSize));
			final Vec2d border = resolution.scale(context.value(this.border));
			Vec2d feather = resolution.scale(context.value(this.feather));
			boolean invert = context.value(this.invert);
			Color color = context.value(this.color);
			double opacity = context.value(this.opacity) / 100;

			final int[] texSize = new int[2];
			tex1d = createGridTextures(texSize, feather, gl);

			gl.glPushAttrib(GL2.GL_CURRENT_BIT | GL2.GL_TEXTURE_BIT);
			try {
				grid = context.createVideoBuffer(bounds);
				VideoEffectUtil.clearTexture(grid, gl);

				final int w = bounds.width;
				final int h = bounds.height;
				VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, grid.getTexture(), 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				final float a = (float)(color.a * (blendMode == BlendMode.NONE ? opacity : 1));
				final float r = (float)color.r * a;
				final float g = (float)color.g * a;
				final float b = (float)color.b * a;
				gl.glColor4f(r, g, b, a);

				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_1D, tex1d[0]);
				gl.glActiveTexture(GL2.GL_TEXTURE1);
				gl.glBindTexture(GL2.GL_TEXTURE_1D, tex1d[1]);

				final IShaderProgram program = invert ? invertSamplerProgram : gridSamplerProgram;
				program.useProgram(new Runnable() {
					public void run() {
						gl.glUniform1i(program.getUniformLocation("texX"), 0);
						gl.glUniform1i(program.getUniformLocation("texY"), 1);

						gl.glUniform4f(program.getUniformLocation("texSize"),
								texSize[0], texSize[1], texSize[0], texSize[1]);

						gl.glUniform4f(program.getUniformLocation("texOffset"),
								(float)((gridSize.x-border.x-texSize[0])*0.5),
								(float)((gridSize.y-border.y-texSize[1])*0.5),
								(float)((gridSize.x+border.x-texSize[0])*0.5),
								(float)((gridSize.y+border.y-texSize[1])*0.5));

						float gridSizeX = Math.max((float)gridSize.x, 1e-10f);
						float gridSizeY = Math.max((float)gridSize.y, 1e-10f);
						gl.glUniform4f(program.getUniformLocation("gridSize"),
								gridSizeX, gridSizeY, gridSizeX, gridSizeY);

						gl.glUniform2f(program.getUniformLocation("anchor"),
								(float)(anchor.x-bounds.x), (float)(anchor.y-bounds.y));

						gl.glUniform4f(program.getUniformLocation("color"), r, g, b, a);

						gl.glBegin(GL2.GL_QUADS);
						gl.glVertex2f(0, 0);
						gl.glVertex2f(w, 0);
						gl.glVertex2f(w, h);
						gl.glVertex2f(0, h);
						gl.glEnd();
					}
				});

				IVideoBuffer result;
				if (blendMode == BlendMode.NONE) {
					result = grid;
					grid = null;
				} else {
					result = blendModeShaders.blend(grid, original, blendMode, opacity);
				}
				return result;

			} finally {
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				gl.glPopAttrib();
			}

		} finally {
			if (original != null) original.dispose();
			if (grid != null) grid.dispose();
			if (tex1d != null) gl.glDeleteTextures(2, tex1d, 0);
		}
	}

	private int[] createGridTextures(int[] texSize, Vec2d feather, GL2 gl) {
		int[] tex1d = null;
		try {
			tex1d = new int[2];
			gl.glGenTextures(2, tex1d, 0);

			texSize[0] = createGridTexture(feather.x, tex1d[0], gl);
			texSize[1] = createGridTexture(feather.y, tex1d[1], gl);

			int[] result = tex1d;
			tex1d = null;
			return result;

		} finally {
			if (tex1d != null) gl.glDeleteTextures(2, tex1d, 0);
		}
	}

	private int createGridTexture(double feather, int tex1d, GL2 gl) {
		IArray<float[]> data = null;
		try {
			int halfLen = (int)Math.ceil(feather) + 1;

			data = arrayPools.getFloatArray(halfLen*2);
			float[] array = data.getArray();
			int arrayLen = data.getLength();

			for (int i = 0; i < arrayLen; ++i) {
				double t = i + 0.5;
				double d = (t <= halfLen-feather) ? 0
						 : (t >= halfLen+feather) ? 1
						 : 0.5*(1-Math.cos(2*Math.PI*(t-(halfLen-feather))/(4*feather)));
				array[i] = (float)d;
			}

			copyTextureData(array, arrayLen, tex1d, gl);
			return arrayLen;

		} finally {
			if (data != null) data.release();
		}
	}

	private void copyTextureData(float[] array, int arrayLen, int tex1d, GL2 gl) {
		// TODO Circleにも同じメソッドがある。

		try {
			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, tex1d);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_EDGE);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_EDGE);

			FloatBuffer buffer = FloatBuffer.wrap(array, 0, arrayLen);

			switch (context.getColorMode()) {
				case RGBA8:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA8, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA16:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA16, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA16_FLOAT:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA16F, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA32_FLOAT:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA32F, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				default:
					throw new RuntimeException("unknown ColorMode: " + context.getColorMode());
			}
		} finally {
			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, 0);
		}
	}

	private static final String[] createSamplerProgram(boolean invert) {
		return new String[] {
				"uniform sampler1D texX;",
				"uniform sampler1D texY;",
				"uniform vec4 texSize;",
				"uniform vec4 texOffset;",
				"uniform vec4 gridSize;",
				"uniform vec2 anchor;",
				"uniform vec4 color;",		// ATIでgl_Colorを使うには頂点シェーダで明示的にglFrontColorを設定する必要がある。
				"",							// ここではそれをせず、uniform変数で色を渡している。
				"void main(void)",
				"{",
				"	vec2 gridCoord = gl_FragCoord.st - anchor;",
				"	gridCoord = gridCoord / gridSize.st + 0.5;",
				"	gridCoord -= floor(gridCoord);",
				"	gridCoord *= gridSize.st;",
				"",
				"	vec4 texCoord = (vec4(gridCoord, gridCoord) - texOffset) / texSize;",
				"",
				"	vec2 a1 = vec2(texture1D(texX, texCoord.x).a, texture1D(texY, texCoord.y).a);",
				"	vec2 a2 = vec2(texture1D(texX, texCoord.z).a, texture1D(texY, texCoord.w).a);",
				"	vec2 a = 1.0 - a1*(1.0-a2);",
				"",
	   invert ? "	gl_FragColor = color * a.x*a.y;"
			  : "	gl_FragColor = color * (1.0-a.x*a.y);",
				"}"
		};
	}

	@ShaderSource
	public static final String[] GRID_SAMPLER = createSamplerProgram(false);

	@ShaderSource
	public static final String[] INVERT_SAMPLER = createSamplerProgram(true);

}
