/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import javax.media.opengl.GL2;
import javax.vecmath.Point2d;
import javax.vecmath.Vector2d;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.BlendMode;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.LinearGradient", category=Effect.GENERATE)
public class LinearGradient extends GradientBase {

	@Property
	private IAnimatableVec2d startPoint;

	@Property
	private IAnimatableVec2d endPoint;

	@Property("1,1,1")
	private IAnimatableColor startColor;

	@Property("0,0,0")
	private IAnimatableColor endColor;

	@Property("RGB")
	private IAnimatableEnum<ColorSpace> colorSpace;

	@Property("CLAMP_TO_EDGE")
	private IAnimatableEnum<RepeatMode> repeatMode;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableEnum<BlendMode> blendMode;


	@Inject
	public LinearGradient(IVideoEffectContext context, IArrayPools arrayPools, IShaderRegistry shaders) {
		super(context, arrayPools, shaders);
	}

	public IVideoBuffer doVideoEffect() {
		return doGradientEffect(context.value(blendMode), context.value(opacity)/100);
	}

	@Override
	protected IVideoBuffer createGradient(VideoBounds bounds) {
		GL2 gl = context.getGL().getGL2();

		IVideoBuffer gradient = null;
		int[] tex1d = null;

		gl.glPushAttrib(GL2.GL_CURRENT_BIT | GL2.GL_TEXTURE_BIT | GL2.GL_ENABLE_BIT);
		try {

			Resolution resolution = context.getVideoResolution();

			Vec2d startPoint = resolution.scale(context.value(this.startPoint));
			Vec2d endPoint = resolution.scale(context.value(this.endPoint));
			Color startColor = context.value(this.startColor);
			Color endColor = context.value(this.endColor);
			ColorSpace colorSpace = context.value(this.colorSpace);
			RepeatMode repeatMode = context.value(this.repeatMode);

			double dx = endPoint.x - startPoint.x;
			double dy = endPoint.y - startPoint.y;
			int texSize = Math.min(4000, (int)Math.ceil(Math.sqrt(dx*dx+dy*dy)));

			if (texSize == 0) {
				IVideoBuffer result = context.createVideoBuffer(bounds);
				VideoEffectUtil.clearTexture(result, gl);
				return result;
			}
			texSize = Math.max(10, texSize);

			switch (colorSpace) {
				case RGB:
					tex1d = createRGBGradientTexture(startColor, endColor, texSize, repeatMode, gl);
					break;
				case HSL:
					tex1d = createHSLGradientTexture(startColor, endColor, false, texSize, repeatMode, gl);
					break;
				case HSL_REVERSE_HUE:
					tex1d = createHSLGradientTexture(startColor, endColor, true, texSize, repeatMode, gl);
					break;
			}

			gradient = context.createVideoBuffer(bounds);
			VideoEffectUtil.clearTexture(gradient, gl);


			double x = bounds.x;
			double y = bounds.y;
			int w = bounds.width;
			int h = bounds.height;

			Point2d center = new Point2d(x+w*0.5, y+h*0.5);
			Vector2d startToEnd = new Vector2d(endPoint.x-startPoint.x, endPoint.y-startPoint.y);

			double theta = Math.atan2(startToEnd.y, startToEnd.x);
			double w2 = w*Math.abs(Math.cos(theta))+h*Math.abs(Math.sin(theta));
			double h2 = h*Math.abs(Math.cos(theta))+w*Math.abs(Math.sin(theta));
			double angle = Math.toDegrees(theta);

			Vector2d startToCenter = new Vector2d(center.x-startPoint.x, center.y-startPoint.y);
			Vector2d endToCenter = new Vector2d(center.x-endPoint.x, center.y-endPoint.y);
			double lenOfStartToEnd = startToEnd.length();
			double coord0 = -(w2*0.5 - startToCenter.dot(startToEnd)/lenOfStartToEnd)/lenOfStartToEnd;
			double coord1 = 1+(w2*0.5 + endToCenter.dot(startToEnd)/lenOfStartToEnd)/lenOfStartToEnd;

			VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);
			gl.glTranslatef((float)(center.x-x), (float)(center.y-y), 0);
			gl.glRotatef((float)angle, 0, 0, 1);
			gl.glScalef((float)(w2/w), (float)(h2/h), 1);
			gl.glTranslatef((float)-(center.x-x), (float)-(center.y-y), 0);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, gradient.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, tex1d[0]);
			gl.glEnable(GL2.GL_TEXTURE_1D);

			gl.glColor4f(1, 1, 1, 1);

			gl.glBegin(GL2.GL_QUADS);
			gl.glTexCoord1f((float)coord0);
			gl.glVertex2f(0, 0);
			gl.glTexCoord1f((float)coord1);
			gl.glVertex2f(w, 0);
			gl.glTexCoord1f((float)coord1);
			gl.glVertex2f(w, h);
			gl.glTexCoord1f((float)coord0);
			gl.glVertex2f(0, h);
			gl.glEnd();

			IVideoBuffer result = gradient;
			gradient = null;
			return result;

		} finally {
			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

			gl.glPopAttrib();

			if (gradient != null) gradient.dispose();
			if (tex1d != null) gl.glDeleteTextures(1, tex1d, 0);
		}
	}

}
