/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.BlendMode;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.RadialGradient", category=Effect.GENERATE)
public class RadialGradient extends GradientBase {

	@Property
	private IAnimatableVec2d center;

	@Property(value="100", min="0", max="4000")
	private IAnimatableDouble radius;

	@Property("1,1,1")
	private IAnimatableColor centerColor;

	@Property("0,0,0")
	private IAnimatableColor edgeColor;

	@Property("RGB")
	private IAnimatableEnum<ColorSpace> colorSpace;

	@Property("CLAMP_TO_EDGE")
	private IAnimatableEnum<RepeatMode> repeatMode;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableEnum<BlendMode> blendMode;


	private final IShaderProgram radialSamplerProgram;

	@Inject
	public RadialGradient(IVideoEffectContext context, IArrayPools arrayPools, IShaderRegistry shaders) {
		super(context, arrayPools, shaders);
		radialSamplerProgram = shaders.getProgram(RadialGradient.class, "RADIAL_SAMPLER");
	}

	public IVideoBuffer doVideoEffect() {
		return doGradientEffect(context.value(blendMode), context.value(opacity)/100);
	}

	@Override
	protected IVideoBuffer createGradient(VideoBounds bounds) {
		final GL2 gl = context.getGL().getGL2();

		IVideoBuffer gradient = null;
		int[] tex1d = null;

		gl.glPushAttrib(GL2.GL_CURRENT_BIT | GL2.GL_TEXTURE_BIT);
		try {

			Resolution resolution = context.getVideoResolution();

			final Vec2d center = resolution.scale(context.value(this.center));
			final double radius = resolution.scale(context.value(this.radius));
			Color centerColor = context.value(this.centerColor);
			Color edgeColor = context.value(this.edgeColor);
			ColorSpace colorSpace = context.value(this.colorSpace);
			RepeatMode repeatMode = context.value(this.repeatMode);

			int texSize = Math.max(10, Math.min(4000, (int)Math.ceil(radius)));

			switch (colorSpace) {
				case RGB:
					tex1d = createRGBGradientTexture(centerColor, edgeColor, texSize, repeatMode, gl);
					break;
				case HSL:
					tex1d = createHSLGradientTexture(centerColor, edgeColor, false, texSize, repeatMode, gl);
					break;
				case HSL_REVERSE_HUE:
					tex1d = createHSLGradientTexture(centerColor, edgeColor, true, texSize, repeatMode, gl);
					break;
			}

			gradient = context.createVideoBuffer(bounds);
			VideoEffectUtil.clearTexture(gradient, gl);


			final double x = bounds.x;
			final double y = bounds.y;
			final int w = bounds.width;
			final int h = bounds.height;

			VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, gradient.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, tex1d[0]);

			gl.glColor4f(1, 1, 1, 1);

			radialSamplerProgram.useProgram(new Runnable() {
				public void run() {
					gl.glUniform1i(radialSamplerProgram.getUniformLocation("texture"), 0);
					gl.glUniform2f(radialSamplerProgram.getUniformLocation("center"), (float)(center.x-x), (float)(center.y-y));
					gl.glUniform1f(radialSamplerProgram.getUniformLocation("radius"), (float)radius);

					gl.glBegin(GL2.GL_QUADS);
					gl.glVertex2f(0, 0);
					gl.glVertex2f(w, 0);
					gl.glVertex2f(w, h);
					gl.glVertex2f(0, h);
					gl.glEnd();
				}
			});

			IVideoBuffer result = gradient;
			gradient = null;
			return result;

		} finally {
			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

			gl.glPopAttrib();

			if (gradient != null) gradient.dispose();
			if (tex1d != null) gl.glDeleteTextures(1, tex1d, 0);
		}
	}

	@ShaderSource
	public static final String[] RADIAL_SAMPLER = {
		"uniform sampler1D texture;",
		"uniform vec2 center;",
		"uniform float radius;",
		"",
		"void main(void)",
		"{",
		"	float d = distance(center, gl_FragCoord.st);",
		"	gl_FragColor = texture1D(texture, d/radius);",
		"}"
	};

}
