/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import java.nio.FloatBuffer;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.BlendMode;
import ch.kuramo.javie.effects.BlendModeShaders;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Rectangle", category=Effect.GENERATE)
public class Rectangle {

	public enum Edge {
		NONE,
		THICKNESS,
		P_THICKNESS,				// PはProportionalのP
		P_THICKNESS_P_FEATHER
	}


	@Property
	private IAnimatableVec2d location;

	@Property(value="150", min="0")
	private IAnimatableVec2d size;

	@Property("0.5")
	private IAnimatableVec2d anchor;

	@Property
	private IAnimatableEnum<Edge> edge;

	@Property(value="10", min="0", max="4000")
	private IAnimatableDouble thickness;

	@Property(min="0", max="4000")
	private IAnimatableDouble outerFeather;

	@Property(min="0", max="4000")
	private IAnimatableDouble innerFeather;

	@Property
	private IAnimatableBoolean invert;

	@Property("1,1,1")
	private IAnimatableColor color;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableEnum<BlendMode> blendMode;


	private final IVideoEffectContext context;

	private final IArrayPools arrayPools;

	private final IShaderProgram rectangleSamplerProgram;

	private final IShaderProgram invertSamplerProgram;

	private final BlendModeShaders blendModeShaders;

	@Inject
	public Rectangle(IVideoEffectContext context, IArrayPools arrayPools, IShaderRegistry shaders) {
		this.context = context;
		this.arrayPools = arrayPools;

		rectangleSamplerProgram = shaders.getProgram(Rectangle.class, "RECTANGLE_SAMPLER");
		invertSamplerProgram = shaders.getProgram(Rectangle.class, "INVERT_SAMPLER");
		blendModeShaders = BlendModeShaders.forPremult(context, shaders);
	}

	public IVideoBuffer doVideoEffect() {
		final GL2 gl = context.getGL().getGL2();

		IVideoBuffer original = null;
		IVideoBuffer rectangle = null;
		int[] tex1d = null;
		try {
			final VideoBounds bounds;

			BlendMode blendMode = context.value(this.blendMode);
			if (blendMode == BlendMode.NONE) {
				bounds = context.getPreviousBounds();
			} else {
				original = context.doPreviousEffect();
				bounds = original.getBounds();
			}

			if (bounds.isEmpty()) {
				if (original == null) {
					original = context.createVideoBuffer(bounds);
					//VideoEffectUtil.clearTexture(original, gl);
					// TODO 空のVideoBufferではあるが、実際には1x1のテクスチャが作成されるのでそれをクリアする必要がある？
					// TODO 他にも同様に空のVideoBufferを作成していてクリアしていない箇所がある。
				}
				IVideoBuffer result = original;
				original = null;
				return result;
			}

			Resolution resolution = context.getVideoResolution();

			Vec2d location = context.value(this.location);
			Vec2d size = context.value(this.size);
			Vec2d anchor = context.value(this.anchor);

			double halfDiag = Math.sqrt(size.x*size.x + size.y*size.y)*0.5;
			final Vec2d center = resolution.scale(new Vec2d(location.x-size.x*(anchor.x-0.5), location.y-size.y*(anchor.y-0.5)));
			size = resolution.scale(size);

			Edge edge = context.value(this.edge);
			boolean invert = context.value(this.invert);
			Color color = context.value(this.color);
			double opacity = context.value(this.opacity) / 100;

			double thickness = 0;
			double innerFeather = 0;
			double outerFeather = 0;

			switch (edge) {
				case NONE:
					outerFeather = resolution.scale(context.value(this.outerFeather));
					break;

				case THICKNESS:
					thickness = resolution.scale(context.value(this.thickness));
					innerFeather = resolution.scale(context.value(this.innerFeather));
					outerFeather = resolution.scale(context.value(this.outerFeather));
					break;

				case P_THICKNESS:
					thickness = resolution.scale(context.value(this.thickness)) * halfDiag / 100;
					innerFeather = resolution.scale(context.value(this.innerFeather));
					outerFeather = resolution.scale(context.value(this.outerFeather));
					break;

				case P_THICKNESS_P_FEATHER:
					thickness = resolution.scale(context.value(this.thickness)) * halfDiag / 100;
					innerFeather = resolution.scale(context.value(this.innerFeather)) * halfDiag / 100;
					outerFeather = resolution.scale(context.value(this.outerFeather)) * halfDiag / 100;
					break;
			}

			final double halfWidth;
			final double halfHeight;
			final double innerMiddleX;
			final double innerMiddleY;
			final double outerMiddleX;
			final double outerMiddleY;
			final int[] texSize = new int[3];

			switch (edge) {
				case NONE:
					halfWidth = size.x*0.5 + outerFeather;
					halfHeight = size.y*0.5 + outerFeather;

					innerMiddleX = 0;
					innerMiddleY = 0;
					outerMiddleX = size.x*0.5;
					outerMiddleY = size.y*0.5;

					tex1d = createRectangleTextures(texSize, 0, 0, 0, outerFeather, gl);
					break;

				default:
					halfWidth = size.x*0.5 + thickness*0.5 + outerFeather;
					halfHeight = size.y*0.5 + thickness*0.5 + outerFeather;

					innerMiddleX = size.x*0.5 - thickness*0.5;
					innerMiddleY = size.y*0.5 - thickness*0.5;
					outerMiddleX = size.x*0.5 + thickness*0.5;
					outerMiddleY = size.y*0.5 + thickness*0.5;

					tex1d = createRectangleTextures(texSize, innerMiddleX, innerMiddleY, innerFeather, outerFeather, gl);
					break;
			}

			gl.glPushAttrib(GL2.GL_CURRENT_BIT | GL2.GL_TEXTURE_BIT);
			try {
				rectangle = context.createVideoBuffer(bounds);
				VideoEffectUtil.clearTexture(rectangle, gl);

				int w = bounds.width;
				int h = bounds.height;
				VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, rectangle.getTexture(), 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				final float a = (float)(color.a * (blendMode == BlendMode.NONE ? opacity : 1));
				final float r = (float)color.r * a;
				final float g = (float)color.g * a;
				final float b = (float)color.b * a;
				gl.glColor4f(r, g, b, a);

				if (invert) {
					gl.glBegin(GL2.GL_QUADS);
					gl.glVertex2f(0, 0);
					gl.glVertex2f(w, 0);
					gl.glVertex2f(w, h);
					gl.glVertex2f(0, h);
					gl.glEnd();
				}

				gl.glTranslatef((float)(center.x-bounds.x), (float)(center.y-bounds.y), 0);

				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_1D, tex1d[0]);
				gl.glActiveTexture(GL2.GL_TEXTURE1);
				gl.glBindTexture(GL2.GL_TEXTURE_1D, tex1d[1]);
				gl.glActiveTexture(GL2.GL_TEXTURE2);
				gl.glBindTexture(GL2.GL_TEXTURE_1D, tex1d[2]);

				final IShaderProgram program = invert ? invertSamplerProgram : rectangleSamplerProgram;
				program.useProgram(new Runnable() {
					public void run() {
						gl.glUniform1i(program.getUniformLocation("texInnerX"), 0);
						gl.glUniform1i(program.getUniformLocation("texInnerY"), 1);
						gl.glUniform1i(program.getUniformLocation("texOuter"), 2);
						gl.glUniform4f(program.getUniformLocation("texSize"),
								texSize[0], texSize[1], texSize[2], texSize[2]);
						gl.glUniform4f(program.getUniformLocation("texOffset"),
								(float)(texSize[0]*0.5-innerMiddleX), (float)(texSize[1]*0.5-innerMiddleY),
								(float)(texSize[2]*0.5-outerMiddleX), (float)(texSize[2]*0.5-outerMiddleY));
						gl.glUniform2f(program.getUniformLocation("center"),
								(float)(center.x-bounds.x), (float)(center.y-bounds.y));
						gl.glUniform4f(program.getUniformLocation("color"), r, g, b, a);

						gl.glBegin(GL2.GL_QUADS);
						gl.glVertex2f((float)-(halfWidth+2), (float)-(halfHeight+2));
						gl.glVertex2f((float) (halfWidth+2), (float)-(halfHeight+2));
						gl.glVertex2f((float) (halfWidth+2), (float) (halfHeight+2));
						gl.glVertex2f((float)-(halfWidth+2), (float) (halfHeight+2));
						gl.glEnd();
					}
				});

				IVideoBuffer result;
				if (blendMode == BlendMode.NONE) {
					result = rectangle;
					rectangle = null;
				} else {
					result = blendModeShaders.blend(rectangle, original, blendMode, opacity);
				}
				return result;

			} finally {
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				gl.glPopAttrib();
			}

		} finally {
			if (original != null) original.dispose();
			if (rectangle != null) rectangle.dispose();
			if (tex1d != null) gl.glDeleteTextures(3, tex1d, 0);
		}
	}

	private int[] createRectangleTextures(int[] texSize,
			double innerMiddleX, double innerMiddleY, double innerFeather, double outerFeather, GL2 gl) {

		int[] tex1d = null;
		try {
			tex1d = new int[3];
			gl.glGenTextures(3, tex1d, 0);

			if (innerMiddleX <= 0 && innerFeather == 0) {
				copyTextureData(new float[] { 0, 0 }, 2, tex1d[0], gl);
				texSize[0] = 2;
			} else {
				texSize[0] = createRectangleTexture(innerFeather, tex1d[0], gl);
			}

			if (innerMiddleY <= 0 && innerFeather == 0) {
				copyTextureData(new float[] { 0, 0 }, 2, tex1d[1], gl);
				texSize[1] = 2;
			} else {
				texSize[1] = createRectangleTexture(innerFeather, tex1d[1], gl);
			}

			texSize[2] = createRectangleTexture(outerFeather, tex1d[2], gl);

			int[] result = tex1d;
			tex1d = null;
			return result;

		} finally {
			if (tex1d != null) gl.glDeleteTextures(3, tex1d, 0);
		}
	}

	private int createRectangleTexture(double feather, int tex1d, GL2 gl) {
		IArray<float[]> data = null;
		try {
			int halfLen = (int)Math.ceil(feather) + 1;

			data = arrayPools.getFloatArray(halfLen*2);
			float[] array = data.getArray();
			int arrayLen = data.getLength();

			for (int i = 0; i < arrayLen; ++i) {
				double r = i + 0.5;
				double d = (r <= halfLen-feather) ? 1
						 : (r >= halfLen+feather) ? 0
						 : 0.5*(1-Math.cos(2*Math.PI*(r-(halfLen+feather))/(4*feather)));

				array[i] = (float) d;
			}

			copyTextureData(array, arrayLen, tex1d, gl);
			return arrayLen;

		} finally {
			if (data != null) data.release();
		}
	}

	private void copyTextureData(float[] array, int arrayLen, int tex1d, GL2 gl) {
		// TODO Circleにも同じメソッドがある。

		try {
			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, tex1d);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_EDGE);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_EDGE);

			FloatBuffer buffer = FloatBuffer.wrap(array, 0, arrayLen);

			switch (context.getColorMode()) {
				case RGBA8:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA8, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA16:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA16, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA16_FLOAT:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA16F, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA32_FLOAT:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA32F, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				default:
					throw new RuntimeException("unknown ColorMode: " + context.getColorMode());
			}
		} finally {
			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, 0);
		}
	}

	private static final String[] createSamplerProgram(boolean invert) {
		return new String[] {
				"uniform sampler1D texInnerX;",
				"uniform sampler1D texInnerY;",
				"uniform sampler1D texOuter;",
				"uniform vec4 texSize;",
				"uniform vec4 texOffset;",
				"uniform vec2 center;",
				"uniform vec4 color;",		// ATIでgl_Colorを使うには頂点シェーダで明示的にglFrontColorを設定する必要がある。
				"",							// ここではそれをせず、uniform変数で色を渡している。
				"void main(void)",
				"{",
				"	vec2 delta = abs(gl_FragCoord.st - center);",
				"	vec4 texCoord = (vec4(delta, delta) + texOffset) / texSize;",
				"",
				"	float a0 = texture1D(texInnerX, texCoord.x).a;",
				"	float a1 = texture1D(texInnerY, texCoord.y).a;",
				"	float a2 = texture1D(texOuter, texCoord.z).a;",
				"	float a3 = texture1D(texOuter, texCoord.w).a;",
				"",
	   invert ? "	gl_FragColor = color * (1.0 - (1.0-a0*a1)*(a2*a3));"
			  : "	gl_FragColor = color * (1.0-a0*a1)*(a2*a3);",
				"}"
		};
	}

	@ShaderSource
	public static final String[] RECTANGLE_SAMPLER = createSamplerProgram(false);

	@ShaderSource
	public static final String[] INVERT_SAMPLER = createSamplerProgram(true);

}
