/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.keying;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.ChromaKey", category=Effect.KEYING)
public class ChromaKey {

	@Property("1,1,1")
	private IAnimatableColor keyColor;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble similarity;

	@Property(value="10", min="0", max="100")
	private IAnimatableDouble blend;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble threshold;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble cutoff;

	@Property
	private IAnimatableEnum<Smoothing> smoothing;

	@Property
	private IAnimatableBoolean maskOnly;


	private final IVideoEffectContext context;

	private final IShaderProgram keyingProgram;

	private final IShaderProgram maskOnlyProgram;

	private final IShaderProgram smoothingLowProgram;

	private final IShaderProgram smoothingHighProgram;

	@Inject
	public ChromaKey(IVideoEffectContext context, IShaderRegistry shaders) {
		this.context = context;

		keyingProgram = shaders.getProgram(KeyingShaders.class, "CHROMA_KEY");
		maskOnlyProgram = shaders.getProgram(KeyingShaders.class, "CHROMA_KEY_MASK_ONLY");
		smoothingLowProgram = shaders.getProgram(KeyingShaders.class, "CHROMA_KEY_SMOOTHING_LOW");
		smoothingHighProgram = shaders.getProgram(KeyingShaders.class, "CHROMA_KEY_SMOOTHING_HIGH");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer input = null;
		IVideoBuffer keyedOut = null;
		IVideoBuffer smoothMask = null;
		try {
			input = context.doPreviousEffect();
			VideoBounds bounds = input.getBounds();
			if (bounds.isEmpty()) {
				IVideoBuffer output = input;
				input = null;
				return output;
			}


			Color keyColor = context.value(this.keyColor);
			final double similarity = context.value(this.similarity) / 100 * 0.4;	// 0.4というのはPEとほぼ同じにするための係数
			final double blend = context.value(this.blend) / 100 * 0.4;
			final double threshold = context.value(this.threshold) / 100;
			final double cutoff = context.value(this.cutoff) / 100;
			Smoothing smoothing = context.value(this.smoothing);
			boolean maskOnly = context.value(this.maskOnly);

			double r = keyColor.r;
			double g = keyColor.g;
			double b = keyColor.b;
			final double u = -0.14713*r -0.28886*g +0.436*b;
			final double v =  0.615*r -0.51499*g -0.10001*b;


			final GL2 gl = context.getGL().getGL2();

			gl.glPushAttrib(GL2.GL_TEXTURE_BIT | GL2.GL_CURRENT_BIT);
			try {
				final int w = bounds.width;
				final int h = bounds.height;
				VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

				keyedOut = context.createVideoBuffer(bounds);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, keyedOut.getTexture(), 0);

				if (smoothing == Smoothing.NONE || maskOnly) {
					gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				} else {
					smoothMask = context.createVideoBuffer(bounds);

					gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
							GL2.GL_COLOR_ATTACHMENT1, GL2.GL_TEXTURE_RECTANGLE, smoothMask.getTexture(), 0);
					gl.glDrawBuffers(2, new int[] { GL2.GL_COLOR_ATTACHMENT0, GL2.GL_COLOR_ATTACHMENT1 }, 0);
				}

				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, input.getTexture());

				gl.glColor4f(1, 1, 1, 1);

				final IShaderProgram program1 = maskOnly ? maskOnlyProgram : keyingProgram;
				program1.useProgram(new Runnable() {
					public void run() {
						gl.glUniform1i(program1.getUniformLocation("texture"), 0);
						gl.glUniform2f(program1.getUniformLocation("keyUV"), (float)u, (float)v);
						gl.glUniform1f(program1.getUniformLocation("similarity"), (float)similarity);
						gl.glUniform1f(program1.getUniformLocation("blend"), (float)blend);
						gl.glUniform1f(program1.getUniformLocation("t_minus_c"), (float)(threshold-cutoff));
						gl.glUniform1f(program1.getUniformLocation("cutoff"), (float)cutoff);

						gl.glBegin(GL2.GL_QUADS);
						gl.glTexCoord2f(0, 0);
						gl.glVertex2f(0, 0);
						gl.glTexCoord2f(w, 0);
						gl.glVertex2f(w, 0);
						gl.glTexCoord2f(w, h);
						gl.glVertex2f(w, h);
						gl.glTexCoord2f(0, h);
						gl.glVertex2f(0, h);
						gl.glEnd();
					}
				});

				if (smoothMask == null) {
					IVideoBuffer output = keyedOut;
					keyedOut = null;
					return output;
				}

				final IShaderProgram program2;
				switch (smoothing) {
					case LOW:
						program2 = smoothingLowProgram;
						break;
					case HIGH:
						program2 = smoothingHighProgram;
						break;
					default:
						throw new Error(); // never reached here!
				}

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, input.getTexture(), 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, keyedOut.getTexture());

				gl.glActiveTexture(GL2.GL_TEXTURE1);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, smoothMask.getTexture());

				program2.useProgram(new Runnable() {
					public void run() {
						gl.glUniform1i(program2.getUniformLocation("tex0"), 0);
						gl.glUniform1i(program2.getUniformLocation("tex1"), 1);

						gl.glBegin(GL2.GL_QUADS);
						gl.glTexCoord2f(0, 0);
						gl.glVertex2f(0, 0);
						gl.glTexCoord2f(w, 0);
						gl.glVertex2f(w, 0);
						gl.glTexCoord2f(w, h);
						gl.glVertex2f(w, h);
						gl.glTexCoord2f(0, h);
						gl.glVertex2f(0, h);
						gl.glEnd();
					}
				});

				IVideoBuffer output = input;
				input = null;
				return output;

			} finally {
				gl.glPopAttrib();

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT1, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
			}

		} finally {
			if (input != null) input.dispose();
			if (keyedOut != null) keyedOut.dispose();
			if (smoothMask != null) smoothMask.dispose();
		}
	}

}
