/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.keying;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.ColorChart", category=Effect.KEYING)
public class ColorChart {

	public enum ChartType { HSL, YUV }


	@Property("HSL")
	private IAnimatableEnum<ChartType> chartType;

	@Property(value="0.5", min="0", max="1.0")
	private IAnimatableDouble luma;

	@Property
	private IAnimatableVec2d location;

	@Property(value="100", min="0")
	private IAnimatableDouble size;


	private final IVideoEffectContext context;

	private final IShaderProgram yuvProgram;

	private final IShaderProgram hslProgram;

	@Inject
	public ColorChart(IVideoEffectContext context, IShaderRegistry shaders) {
		this.context = context;
		yuvProgram = shaders.getProgram(ColorChart.class, "YUV_CHART");
		hslProgram = shaders.getProgram(ColorChart.class, "HSL_CHART");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer input = null;
		try {
			input = context.doPreviousEffect();
			VideoBounds bounds = input.getBounds();
			if (bounds.isEmpty()) {
				IVideoBuffer output = input;
				input = null;
				return output;
			}


			ChartType chartType = context.value(this.chartType);
			double luma = context.value(this.luma);

			Resolution resolution = context.getVideoResolution();
			Vec2d location = resolution.scale(context.value(this.location));
			double size = resolution.scale(context.value(this.size));


			GL2 gl = context.getGL().getGL2();

			gl.glPushAttrib(GL2.GL_CURRENT_BIT);
			try {
				int w = bounds.width;
				int h = bounds.height;
				VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, input.getTexture(), 0);

				gl.glColor4f(1, 1, 1, 1);

				switch (chartType) {
					case YUV:
						drawYUVChart(location, size, luma, gl);
						break;
					case HSL:
						drawHSLChart(location, size, luma, gl);
						break;
				}

			} finally {
				gl.glPopAttrib();

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
			}

			IVideoBuffer result = input;
			input = null;
			return result;

		} finally {
			if (input != null) input.dispose();
		}
	}

	private void drawYUVChart(final Vec2d location, final double size, final double luma, final GL2 gl) {
		yuvProgram.useProgram(new Runnable() {
			public void run() {
				gl.glUniform2f(yuvProgram.getUniformLocation("location"), (float)location.x, (float)location.y);
				gl.glUniform1f(yuvProgram.getUniformLocation("size"), (float)size);
				gl.glUniform1f(yuvProgram.getUniformLocation("luma"), (float)luma);

				gl.glBegin(GL2.GL_QUADS);
				gl.glVertex2f((float)location.x       , (float)location.y       );
				gl.glVertex2f((float)(location.x+size), (float)location.y       );
				gl.glVertex2f((float)(location.x+size), (float)(location.y+size));
				gl.glVertex2f((float)location.x       , (float)(location.y+size));
				gl.glEnd();
			}
		});
	}

	@ShaderSource
	public static final String[] YUV_CHART = {
		"uniform vec2 location;",
		"uniform float size;",
		"uniform float luma;",
		"",
		"",	// Vの項の符号が通常とは逆なのは、Y座標の向きが逆なため。
		"const vec3 rVec = vec3(1.0,  0.0    , -1.13983);",
		"const vec3 gVec = vec3(1.0, -0.39465,  0.58060);",
		"const vec3 bVec = vec3(1.0,  2.03211,  0.0);",
		"",
		"const vec3 vec3_000 = vec3(0.0);",
		"const vec3 vec3_111 = vec3(1.0);",
		"",
		"void main(void)",
		"{",
		"	vec3 yuv = vec3(luma, (gl_FragCoord.xy-location)/size-0.5);",
		"	vec3 rgb = vec3(dot(yuv, rVec), dot(yuv, gVec), dot(yuv, bVec));",
		"",
		"	if (all(greaterThanEqual(rgb, vec3_000)) && all(lessThanEqual(rgb, vec3_111))) {",
		"		gl_FragColor = vec4(rgb, 1.0);",
		"	} else {",
		"		gl_FragColor = vec4(0.5, 0.5, 0.5, 1.0);",
		"	}",
		"}",
	};

	private void drawHSLChart(final Vec2d location, final double size, final double luma, final GL2 gl) {
		hslProgram.useProgram(new Runnable() {
			public void run() {
				double radius = size*0.5;
				gl.glUniform2f(hslProgram.getUniformLocation("center"), (float)(location.x+radius), (float)(location.y+radius));
				gl.glUniform1f(hslProgram.getUniformLocation("radius"), (float)radius);
				gl.glUniform1f(hslProgram.getUniformLocation("luma"), (float)luma);

				gl.glBegin(GL2.GL_QUADS);
				gl.glVertex2f((float)location.x       , (float)location.y       );
				gl.glVertex2f((float)(location.x+size), (float)location.y       );
				gl.glVertex2f((float)(location.x+size), (float)(location.y+size));
				gl.glVertex2f((float)location.x       , (float)(location.y+size));
				gl.glEnd();
			}
		});
	}

	// TODO FractalNoiseにあるhsl2rgbをどこか適切な場所に移動する。
	@ShaderSource(attach="ch.kuramo.javie.effects.noiseGrain.FractalNoise.hsl2rgb")
	public static final String[] HSL_CHART = {
		"uniform vec2 center;",
		"uniform float radius;",
		"uniform float luma;",
		"",
		"const float PI2 = 2.0*3.14159265358979323846264;",
		"vec3 hsl2rgb(vec3 hsl);",
		"",
		"void main(void)",
		"{",
		"	vec2 xy = gl_FragCoord.xy-center;",
		"	float h = -0.25-atan(xy.y, xy.x)/PI2;",
		"	float s = distance(gl_FragCoord.xy, center)/radius;",
		"	if (s <= 1.0) {",
		"		vec3 hsl = vec3(h-floor(h), s, luma);",
		"		vec3 rgb = hsl2rgb(hsl);",
		"		gl_FragColor = vec4(rgb, 1.0);",
		"	} else {",
		"		gl_FragColor = vec4(0.5, 0.5, 0.5, 1.0);",
		"	}",
		"}",
	};

}
