/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.keying;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;
import javax.media.opengl.glu.GLUtessellator;
import javax.media.opengl.glu.GLUtessellatorCallbackAdapter;

import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IAntiAliasSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.VideoEffectUtil;
import ch.kuramo.javie.effects.blurSharpen.BlurUtil;

public class GarbageMatteBase {

	protected final IVideoEffectContext context;

	protected final IAntiAliasSupport aaSupport;

	protected final IShaderRegistry shaders;


	protected GarbageMatteBase(IVideoEffectContext context, IAntiAliasSupport aaSupport, IShaderRegistry shaders) {
		this.context = context;
		this.aaSupport = aaSupport;
		this.shaders = shaders;
	}

	protected IVideoBuffer doMask(final boolean invert, final Vec2d ... points) {
		IVideoBuffer src = null;
		IVideoBuffer dst = null;
		GLUtessellator tessellator = null;
		try {
			src = context.doPreviousEffect();

			VideoBounds bounds = src.getBounds();
			if (bounds.isEmpty()) {
				IVideoBuffer result = src;
				src = null;
				return result;
			}

			GL2 gl = context.getGL().getGL2();
			GLU glu = context.getGLU();

			dst = context.createVideoBuffer(bounds);
			VideoEffectUtil.clearTexture(dst, gl);

			tessellator = GLU.gluNewTess();
			TessellatorCallback callback = new TessellatorCallback(-bounds.x, -bounds.y, gl, glu);
			GLU.gluTessCallback(tessellator, GLU.GLU_TESS_BEGIN, callback);
			GLU.gluTessCallback(tessellator, GLU.GLU_TESS_END, callback);
			GLU.gluTessCallback(tessellator, GLU.GLU_TESS_VERTEX, callback);
			GLU.gluTessCallback(tessellator, GLU.GLU_TESS_COMBINE, callback);
			GLU.gluTessCallback(tessellator, GLU.GLU_TESS_ERROR, callback);


			gl.glPushAttrib(GL2.GL_ENABLE_BIT | GL2.GL_TEXTURE_BIT | GL2.GL_CURRENT_BIT);
			try {
				final double x = bounds.x;
				final double y = bounds.y;
				final int w = bounds.width;
				final int h = bounds.height;

				VideoEffectUtil.ortho2D(gl, glu, w, h);
				gl.glTranslatef((float)-x, (float)-y, 0);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, dst.getTexture(), 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, src.getTexture());
				gl.glEnable(GL2.GL_TEXTURE_RECTANGLE);

				gl.glColor4f(1, 1, 1, 1);

				final GLUtessellator tess = tessellator;
				aaSupport.antiAlias(w, h, new Runnable() {
					public void run() {
						GLU.gluTessBeginPolygon(tess, null);
						if (invert) {
							GLU.gluTessBeginContour(tess);
							for (double[] vertex : new double[][] { {x,y,0}, {x+w,y,0}, {x+w,y+h,0}, {x,y+h,0} }) {
								GLU.gluTessVertex(tess, vertex, 0, vertex);
							}
							GLU.gluTessEndContour(tess);
						}
						GLU.gluTessBeginContour(tess);
						for (Vec2d pt : points) {
							double[] vertex = new double[] { pt.x, pt.y, 0 };
							GLU.gluTessVertex(tess, vertex, 0, vertex);
						}
						GLU.gluTessEndContour(tess);
						GLU.gluTessEndPolygon(tess);
					}
				});

			} finally {
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				gl.glPopAttrib();
			}

			IVideoBuffer result = dst;
			dst = null;
			return result;

		} finally {
			if (tessellator != null) GLU.gluDeleteTess(tessellator);
			if (dst != null) dst.dispose();
			if (src != null) src.dispose();
		}
	}

	protected IVideoBuffer doMask(final boolean invert, double feather, final Vec2d ... points) {
		IVideoBuffer src = null;
		IVideoBuffer mask = null;
		GLUtessellator tessellator = null;
		try {
			src = context.doPreviousEffect();

			VideoBounds bounds = src.getBounds();
			if (bounds.isEmpty()) {
				IVideoBuffer result = src;
				src = null;
				return result;
			}

			GL2 gl = context.getGL().getGL2();
			GLU glu = context.getGLU();

			mask = context.createVideoBuffer(bounds);
			VideoEffectUtil.clearTexture(mask, gl);

			tessellator = GLU.gluNewTess();
			TessellatorCallback callback = new TessellatorCallback(gl, glu);
			GLU.gluTessCallback(tessellator, GLU.GLU_TESS_BEGIN, callback);
			GLU.gluTessCallback(tessellator, GLU.GLU_TESS_END, callback);
			GLU.gluTessCallback(tessellator, GLU.GLU_TESS_VERTEX, callback);
			GLU.gluTessCallback(tessellator, GLU.GLU_TESS_COMBINE, callback);
			GLU.gluTessCallback(tessellator, GLU.GLU_TESS_ERROR, callback);


			gl.glPushAttrib(GL2.GL_ENABLE_BIT | GL2.GL_TEXTURE_BIT | GL2.GL_CURRENT_BIT | GL2.GL_COLOR_BUFFER_BIT);
			try {
				final double x = bounds.x;
				final double y = bounds.y;
				final int w = bounds.width;
				final int h = bounds.height;

				VideoEffectUtil.ortho2D(gl, glu, w, h);
				gl.glTranslatef((float)-x, (float)-y, 0);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, mask.getTexture(), 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				gl.glColor4f(1, 1, 1, 1);

				final GLUtessellator tess = tessellator;
				aaSupport.antiAlias(w, h, new Runnable() {
					public void run() {
						GLU.gluTessBeginPolygon(tess, null);
						if (invert) {
							GLU.gluTessBeginContour(tess);
							for (double[] vertex : new double[][] { {x,y,0}, {x+w,y,0}, {x+w,y+h,0}, {x,y+h,0} }) {
								GLU.gluTessVertex(tess, vertex, 0, vertex);
							}
							GLU.gluTessEndContour(tess);
						}
						GLU.gluTessBeginContour(tess);
						for (Vec2d pt : points) {
							double[] vertex = new double[] { pt.x, pt.y, 0 };
							GLU.gluTessVertex(tess, vertex, 0, vertex);
						}
						GLU.gluTessEndContour(tess);
						GLU.gluTessEndPolygon(tess);
					}
				});

				mask = blur(mask, feather, gl, glu);

				VideoEffectUtil.ortho2D(gl, glu, w, h);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, src.getTexture(), 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, mask.getTexture());
				gl.glEnable(GL2.GL_TEXTURE_RECTANGLE);

				gl.glEnable(GL2.GL_BLEND);
				gl.glBlendFuncSeparate(GL2.GL_ZERO, GL2.GL_SRC_ALPHA, GL2.GL_ZERO, GL2.GL_SRC_ALPHA);
				gl.glBlendEquation(GL2.GL_FUNC_ADD);

				gl.glBegin(GL2.GL_QUADS);
				gl.glTexCoord2f(0, 0);
				gl.glVertex2f(0, 0);
				gl.glTexCoord2f(w, 0);
				gl.glVertex2f(w, 0);
				gl.glTexCoord2f(w, h);
				gl.glVertex2f(w, h);
				gl.glTexCoord2f(0, h);
				gl.glVertex2f(0, h);
				gl.glEnd();

			} finally {
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				gl.glPopAttrib();
			}

			IVideoBuffer result = src;
			src = null;
			return result;

		} finally {
			if (tessellator != null) GLU.gluDeleteTess(tessellator);
			if (mask != null) mask.dispose();
			if (src != null) src.dispose();
		}
	}

	private static class TessellatorCallback extends GLUtessellatorCallbackAdapter {

		private final double[] texOffset;

		private final GL2 gl;

		private final GLU glu;


		private TessellatorCallback(GL2 gl, GLU glu) {
			this(null, gl, glu);
		}

		private TessellatorCallback(double texOffsetX, double texOffsetY, GL2 gl, GLU glu) {
			this(new double[] { texOffsetX, texOffsetY }, gl, glu);
		}

		private TessellatorCallback(double[] texOffset, GL2 gl, GLU glu) {
			this.texOffset = texOffset;
			this.gl = gl;
			this.glu = glu;
		}

		@Override
		public void begin(int type) {
			gl.glBegin(type);
		}

		@Override
		public void end() {
			gl.glEnd();
		}

		@Override
		public void vertex(Object vertexData) {
			double[] vertex = (double[]) vertexData;
			if (texOffset != null) {
				gl.glTexCoord2d(vertex[0]+texOffset[0], vertex[1]+texOffset[1]);
			}
			gl.glVertex2dv(vertex, 0);
		}

		@Override
		public void combine(double[] coords, Object[] data, float[] weight, Object[] outData) {
			outData[0] = new double[] { coords[0], coords[1], coords[2] };
		}

		@Override
		public void error(int errnum) {
			System.err.println("Tessellation Error: " + glu.gluErrorString(errnum));
		}

	};

	private IVideoBuffer blur(IVideoBuffer input, double blur, GL2 gl, GLU glu) {
		return blur(input, blur, true, true, true, true, gl, glu);
	}

	private IVideoBuffer blur(
			IVideoBuffer input, double blur, boolean repeatEdgePixels,
			boolean horz, boolean vert, boolean fast, GL2 gl, GLU glu) {

		// TODO 以下は GaussianBlur から丸々コピペしたもの。整理する。
		// TODO ぼかすのはアルファチャンネルだけでよい。
		// TODO ぼかすのは境界線付近だけでよい。

		VideoBounds inputBounds = input.getBounds();

		int sampleRatio = 1;
		if (fast) {
			VideoBounds bounds = inputBounds;
			for (int factor : BlurUtil.getDownSampleFactors(blur)) {
				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(input, gl);
				}

				int hFactor = horz ? factor : 1;
				int vFactor = vert ? factor : 1;
				bounds = new VideoBounds(bounds.x/hFactor, bounds.y/vFactor,
						(bounds.width+hFactor-1)/hFactor, (bounds.height+vFactor-1)/vFactor);

				IVideoBuffer buf = context.createVideoBuffer(bounds);
				BlurUtil.doDownSample(input, buf, hFactor, vFactor, gl, glu, shaders);
				input.dispose();
				input = buf;

				sampleRatio *= factor;
			}
			blur /= sampleRatio;
		}

		int radius = (int) Math.ceil(blur);


		// 標準偏差の2.5倍の位置がぼかしの端となるようにする（これでだいたいAEと同じになる）
		double sigma = blur / 2.5;
		double sigmaSquare = sigma * sigma;

		float[] kernel = new float[radius*2+1];
		float sum = 0;
		for (int i = 1; i <= radius; ++i) {
			sum += 2 * (kernel[radius-i] = (float) Math.exp(-i * i / (2 * sigmaSquare)));
		}
		kernel[radius] = 1 / (++sum);
		for (int i = 1; i <= radius; ++i) {
			kernel[radius+i] = (kernel[radius-i] /= sum);
		}


		IVideoBuffer buf1 = null, buf2 = null, buf3 = null;
		try {
			if (horz) {
				VideoBounds bounds = input.getBounds();

				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(input, gl);
				} else {
					bounds = VideoEffectUtil.expandBounds(bounds, radius, true, false);
				}

				buf1 = context.createVideoBuffer(bounds);
				VideoEffectUtil.convolution1D(input, buf1, true, radius, kernel, gl, glu, shaders);
			} else {
				buf1 = input;
				input = null;
			}

			if (vert) {
				VideoBounds bounds = buf1.getBounds();

				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(buf1, gl);
				} else {
					bounds = VideoEffectUtil.expandBounds(bounds, radius, false, true);
				}

				buf2 = context.createVideoBuffer(bounds);
				VideoEffectUtil.convolution1D(buf1, buf2, false, radius, kernel, gl, glu, shaders);
			} else {
				buf2 = buf1;
				buf1 = null;
			}

			if (sampleRatio != 1) {
				int hRatio = horz ? sampleRatio : 1;
				int vRatio = vert ? sampleRatio : 1;
				VideoBounds bounds;

				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(buf2, gl);
					bounds = inputBounds;
				} else {
					bounds = buf2.getBounds();
					bounds = new VideoBounds(bounds.x*hRatio, bounds.y*vRatio, bounds.width*hRatio, bounds.height*vRatio);
				}

				buf3 = context.createVideoBuffer(bounds);
				BlurUtil.doUpSample(buf2, buf3, hRatio, vRatio, gl, glu);
			} else {
				buf3 = buf2;
				buf2 = null;
			}

			return buf3;

		} finally {
			if (input != null) {
				input.dispose();
			}
			if (buf1 != null) {
				buf1.dispose();
			}
			if (buf2 != null) {
				buf2.dispose();
			}
		}
	}

}
