/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.keying;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.HSLKey", category=Effect.KEYING)
public class HSLKey {

	@Property("1,1,1")
	private IAnimatableColor keyColor;

	@Property(value="10", min="0", max="360")
	private IAnimatableVec2d similarityHue;

	@Property(value="10", min="0", max="200")
	private IAnimatableVec2d similaritySat;

	@Property(value="10", min="0", max="100")
	private IAnimatableVec2d similarityLuma;

	@Property(value="10", min="0", max="360")
	private IAnimatableVec2d blendHue;

	@Property(value="10", min="0", max="200")
	private IAnimatableVec2d blendSat;

	@Property(value="10", min="0", max="100")
	private IAnimatableVec2d blendLuma;

	@Property(value="1", min="0", max="3")
	private IAnimatableVec2d correctionHue;

	@Property(value="1", min="0", max="2")
	private IAnimatableDouble correctionSat;

	@Property(value="1", min="0", max="1")
	private IAnimatableDouble correctionAlpha;

	@Property
	private IAnimatableEnum<Smoothing> smoothing;

	@Property
	private IAnimatableBoolean maskOnly;


	private final IVideoEffectContext context;

	private final IShaderProgram keyingProgram;

	private final IShaderProgram maskOnlyProgram;

	private final IShaderProgram smoothingLowProgram;

	private final IShaderProgram smoothingHighProgram;

	@Inject
	public HSLKey(IVideoEffectContext context, IShaderRegistry shaders) {
		this.context = context;

		keyingProgram = shaders.getProgram(KeyingShaders.class, "HSL_KEY");
		maskOnlyProgram = shaders.getProgram(KeyingShaders.class, "HSL_KEY_MASK_ONLY");
		smoothingLowProgram = shaders.getProgram(KeyingShaders.class, "SMOOTHING_LOW");
		smoothingHighProgram = shaders.getProgram(KeyingShaders.class, "SMOOTHING_HIGH");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer input = null;
		IVideoBuffer keyedOut = null;
		try {
			input = context.doPreviousEffect();
			VideoBounds bounds = input.getBounds();
			if (bounds.isEmpty()) {
				IVideoBuffer output = input;
				input = null;
				return output;
			}


			final Color keyColor = context.value(this.keyColor);
			final Vec2d similarityHue = context.value(this.similarityHue);
			final Vec2d similaritySat = context.value(this.similaritySat);
			final Vec2d similarityLuma = context.value(this.similarityLuma);
			final Vec2d blendHue = context.value(this.blendHue);
			final Vec2d blendSat = context.value(this.blendSat);
			final Vec2d blendLuma = context.value(this.blendLuma);
			final Vec2d correctionHue = context.value(this.correctionHue);
			final double correctionSat = context.value(this.correctionSat);
			final double correctionAlpha = context.value(this.correctionAlpha);
			Smoothing smoothing = context.value(this.smoothing);
			final boolean maskOnly = context.value(this.maskOnly);


			final GL2 gl = context.getGL().getGL2();

			gl.glPushAttrib(GL2.GL_TEXTURE_BIT | GL2.GL_CURRENT_BIT);
			try {
				final int w = bounds.width;
				final int h = bounds.height;
				VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

				keyedOut = context.createVideoBuffer(bounds);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, keyedOut.getTexture(), 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, input.getTexture());

				gl.glColor4f(1, 1, 1, 1);

				final IShaderProgram program = maskOnly ? maskOnlyProgram : keyingProgram;
				program.useProgram(new Runnable() {
					public void run() {
						double[] keyHSL = new double[3];
						rgb2hsl(keyColor, keyHSL);

						double hueMin = keyHSL[0] - similarityHue.x/360;
						double hueMax = keyHSL[0] + similarityHue.y/360;
						double satMin = keyHSL[1] - similaritySat.x/100;
						double satMax = keyHSL[1] + similaritySat.y/100;
						double lumaMin = keyHSL[2] - similarityLuma.x/100;
						double lumaMax = keyHSL[2] + similarityLuma.y/100;

						double blendHueMin = Math.max(blendHue.x/360, 1e-10f);
						double blendHueMax = Math.max(blendHue.y/360, 1e-10f);
						double blendSatMin = Math.max(blendSat.x/100, 1e-10f);
						double blendSatMax = Math.max(blendSat.y/100, 1e-10f);
						double blendLumaMin = Math.max(blendLuma.x/100, 1e-10f);
						double blendLumaMax = Math.max(blendLuma.y/100, 1e-10f);

						gl.glUniform1i(program.getUniformLocation("texture"), 0);
						gl.glUniform3f(program.getUniformLocation("similarityMin"), (float)hueMin, (float)satMin, (float)lumaMin);
						gl.glUniform3f(program.getUniformLocation("similarityMax"), (float)hueMax, (float)satMax, (float)lumaMax);
						gl.glUniform3f(program.getUniformLocation("blendMin"), (float)blendHueMin, (float)blendSatMin, (float)blendLumaMin);
						gl.glUniform3f(program.getUniformLocation("blendMax"), (float)blendHueMax, (float)blendSatMax, (float)blendLumaMax);

						double[] bleachHSL = { keyHSL[0], 1.0, 0.5 };
						double[] bleachRGB = new double[3];
						hsl2rgb(bleachHSL, bleachRGB);

						gl.glUniform3f(program.getUniformLocation("bleachHSL"), (float)bleachHSL[0], (float)bleachHSL[1], (float)bleachHSL[2]);
						gl.glUniform3f(program.getUniformLocation("bleachRGB"), (float)bleachRGB[0], (float)bleachRGB[1], (float)bleachRGB[2]);

						gl.glUniform2f(program.getUniformLocation("correctionHue"), (float)correctionHue.x, (float)correctionHue.y);
						gl.glUniform1f(program.getUniformLocation("correctionSat"), (float)(2-correctionSat));
						gl.glUniform1f(program.getUniformLocation("correctionAlpha"), (float)correctionAlpha);

						gl.glBegin(GL2.GL_QUADS);
						gl.glTexCoord2f(0, 0);
						gl.glVertex2f(0, 0);
						gl.glTexCoord2f(w, 0);
						gl.glVertex2f(w, 0);
						gl.glTexCoord2f(w, h);
						gl.glVertex2f(w, h);
						gl.glTexCoord2f(0, h);
						gl.glVertex2f(0, h);
						gl.glEnd();
					}
				});

				if (smoothing == Smoothing.NONE || maskOnly) {
					IVideoBuffer output = keyedOut;
					keyedOut = null;
					return output;
				}

				final IShaderProgram smoothingProgram;
				switch (smoothing) {
					case LOW:
						smoothingProgram = smoothingLowProgram;
						break;
					case HIGH:
						smoothingProgram = smoothingHighProgram;
						break;
					default:
						throw new Error(); // never reached here!
				}

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, input.getTexture(), 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, keyedOut.getTexture());

				smoothingProgram.useProgram(new Runnable() {
					public void run() {
						gl.glUniform1i(smoothingProgram.getUniformLocation("texture"), 0);

						gl.glBegin(GL2.GL_QUADS);
						gl.glTexCoord2f(0, 0);
						gl.glVertex2f(0, 0);
						gl.glTexCoord2f(w, 0);
						gl.glVertex2f(w, 0);
						gl.glTexCoord2f(w, h);
						gl.glVertex2f(w, h);
						gl.glTexCoord2f(0, h);
						gl.glVertex2f(0, h);
						gl.glEnd();
					}
				});

				IVideoBuffer output = input;
				input = null;
				return output;

			} finally {
				gl.glPopAttrib();

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
			}

		} finally {
			if (input != null) input.dispose();
			if (keyedOut != null) keyedOut.dispose();
		}
	}

	// TODO 以下、GradientBaseに同じものがある。

	private void rgb2hsl(Color rgb, double[] hsl) {
		double min = Math.min(Math.min(rgb.r, rgb.g), rgb.b);
		double max = Math.max(Math.max(rgb.r, rgb.g), rgb.b);
		double dmax = max - min;

		double luma = (max + min)*0.5;
		double hue, sat;

		if (dmax == 0) {
			hue = sat = 0;
		} else {
			sat = (luma < 0.5) ? dmax/(max+min) : dmax/(2-max-min);

			double dr = ((max-rgb.r)/6 + dmax/2)/dmax;
			double dg = ((max-rgb.g)/6 + dmax/2)/dmax;
			double db = ((max-rgb.b)/6 + dmax/2)/dmax;

			hue = (rgb.r == max) ? db-dg
				: (rgb.g == max) ? 1./3 + dr-db
				:				   2./3 + dg-dr;

			if (hue < 0) hue += 1;
			else if (hue > 1) hue -= 1;
		}

		hsl[0] = hue;
		hsl[1] = sat;
		hsl[2] = luma;
	}

	private void hsl2rgb(double[] hsl, double[] rgb) {
		double hue = hsl[0];
		double sat = hsl[1];
		double luma = hsl[2];

		if (hue > 1) hue -= 1;

		if (sat == 0) {
			rgb[0] = rgb[1] = rgb[2] = luma;
		} else {
			double t2 = (luma < 0.5) ? luma*(1+sat) : luma+sat-luma*sat;
			double t1 = luma*2 - t2;

			rgb[0] = hue2rgb(t1, t2, hue+1./3);
			rgb[1] = hue2rgb(t1, t2, hue);
			rgb[2] = hue2rgb(t1, t2, hue-1./3);
		}
	}

	private double hue2rgb(double t1, double t2, double hue) {
		if (hue < 0) hue += 1;
		else if (hue > 1) hue -= 1;

		return (hue*6 < 1) ? t1 + (t2-t1)*6*hue
			 : (hue*2 < 1) ? t2
			 : (hue*3 < 2) ? t1 + (t2-t1)*(2./3-hue)*6
			 :				 t1;
	}

}
