/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.keying;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import ch.kuramo.javie.api.annotations.ShaderSource;

public class KeyingShaders {

	@ShaderSource
	public static final String[] BLUE_SCREEN_KEY = {
		"uniform sampler2DRect texture;",
		"uniform float threshold;",
		"uniform float t_minus_c;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
		"	vec3 rgb = color.rgb/color.a;",		// color.a == 0.0 の場合でも残りの計算に影響はない。
		"	color.a = min(color.a, color.a*(threshold-rgb.b*(1.0-rgb.r))/t_minus_c);",
		"	rgb.b = min(rgb.b, rgb.g);",
		"	gl_FragColor = vec4(rgb, 1.0)*color.a;",
		"}"
	};

	@ShaderSource
	public static final String[] BLUE_SCREEN_KEY_MASK_ONLY = {
		"uniform sampler2DRect texture;",
		"uniform float threshold;",
		"uniform float t_minus_c;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
		"	vec3 rgb = color.rgb/color.a;",
		"	float a = min(color.a, color.a*(threshold-rgb.b*(1.0-rgb.r))/t_minus_c);",
		"	gl_FragColor = vec4(a, a, a, 1.0);",
		"}"
	};

	@ShaderSource
	public static final String[] GREEN_SCREEN_KEY = {
		"uniform sampler2DRect texture;",
		"uniform float threshold;",
		"uniform float t_minus_c;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
		"	vec3 rgb = color.rgb/color.a;",
		"	color.a = min(color.a, color.a*(threshold-rgb.g*(1.0-rgb.r))/t_minus_c);",
		"	rgb.g = min(rgb.g, rgb.r);",
		"	gl_FragColor = vec4(rgb, 1.0)*color.a;",
		"}"
	};

	@ShaderSource
	public static final String[] GREEN_SCREEN_KEY_MASK_ONLY = {
		"uniform sampler2DRect texture;",
		"uniform float threshold;",
		"uniform float t_minus_c;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
		"	vec3 rgb = color.rgb/color.a;",
		"	float a = min(color.a, color.a*(threshold-rgb.g*(1.0-rgb.r))/t_minus_c);",
		"	gl_FragColor = vec4(a, a, a, 1.0);",
		"}"
	};

	@ShaderSource
	public static final String[] SMOOTHING_LOW = {
		"uniform sampler2DRect texture;",
		"",
		"void main(void)",
		"{",
		"	vec2 coord = gl_TexCoord[0].st;",
		"	vec4 sum = vec4(0.0);",
		"	int n = 0;",
		"	vec4 c;",
		"	c = texture2DRect(texture, coord+vec2(-1.0, -1.0)); if (c.a != 0.0) { ++n; sum += 0.0625 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 0.0, -1.0)); if (c.a != 0.0) { ++n; sum += 0.1250 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 1.0, -1.0)); if (c.a != 0.0) { ++n; sum += 0.0625 * c; }",
		"	c = texture2DRect(texture, coord+vec2(-1.0,  0.0)); if (c.a != 0.0) { ++n; sum += 0.1250 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 1.0,  0.0)); if (c.a != 0.0) { ++n; sum += 0.1250 * c; }",
		"	c = texture2DRect(texture, coord+vec2(-1.0,  1.0)); if (c.a != 0.0) { ++n; sum += 0.0625 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 0.0,  1.0)); if (c.a != 0.0) { ++n; sum += 0.1250 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 1.0,  1.0)); if (c.a != 0.0) { ++n; sum += 0.0625 * c; }",
		"	c = texture2DRect(texture, coord                 ); if (c.a != 0.0) { ++n; sum += 0.2500 * c; }",
		"",
		"	gl_FragColor = (n > 0 && n < 9) ? sum : c;",
		"}"
	};

	@ShaderSource
	public static final String[] SMOOTHING_HIGH = {
		"uniform sampler2DRect texture;",
		"",
		"void main(void)",
		"{",
		"	vec2 coord = gl_TexCoord[0].st;",
		"	vec4 sum = vec4(0.0);",
		"	int n = 0;",
		"	vec4 c;",
		"	c = texture2DRect(texture, coord+vec2(-2.0, -2.0)); if (c.a != 0.0) { ++n; sum += 0.003663 * c; }",
		"	c = texture2DRect(texture, coord+vec2(-1.0, -2.0)); if (c.a != 0.0) { ++n; sum += 0.014652 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 0.0, -2.0)); if (c.a != 0.0) { ++n; sum += 0.025641 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 1.0, -2.0)); if (c.a != 0.0) { ++n; sum += 0.014652 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 2.0, -2.0)); if (c.a != 0.0) { ++n; sum += 0.003663 * c; }",
		"	c = texture2DRect(texture, coord+vec2(-2.0, -1.0)); if (c.a != 0.0) { ++n; sum += 0.014652 * c; }",
		"	c = texture2DRect(texture, coord+vec2(-1.0, -1.0)); if (c.a != 0.0) { ++n; sum += 0.058608 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 0.0, -1.0)); if (c.a != 0.0) { ++n; sum += 0.095238 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 1.0, -1.0)); if (c.a != 0.0) { ++n; sum += 0.058608 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 2.0, -1.0)); if (c.a != 0.0) { ++n; sum += 0.014652 * c; }",
		"	c = texture2DRect(texture, coord+vec2(-2.0,  0.0)); if (c.a != 0.0) { ++n; sum += 0.025641 * c; }",
		"	c = texture2DRect(texture, coord+vec2(-1.0,  0.0)); if (c.a != 0.0) { ++n; sum += 0.095238 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 1.0,  0.0)); if (c.a != 0.0) { ++n; sum += 0.095238 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 2.0,  0.0)); if (c.a != 0.0) { ++n; sum += 0.025641 * c; }",
		"	c = texture2DRect(texture, coord+vec2(-2.0,  1.0)); if (c.a != 0.0) { ++n; sum += 0.014652 * c; }",
		"	c = texture2DRect(texture, coord+vec2(-1.0,  1.0)); if (c.a != 0.0) { ++n; sum += 0.058608 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 0.0,  1.0)); if (c.a != 0.0) { ++n; sum += 0.095238 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 1.0,  1.0)); if (c.a != 0.0) { ++n; sum += 0.058608 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 2.0,  1.0)); if (c.a != 0.0) { ++n; sum += 0.014652 * c; }",
		"	c = texture2DRect(texture, coord+vec2(-2.0,  2.0)); if (c.a != 0.0) { ++n; sum += 0.003663 * c; }",
		"	c = texture2DRect(texture, coord+vec2(-1.0,  2.0)); if (c.a != 0.0) { ++n; sum += 0.014652 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 0.0,  2.0)); if (c.a != 0.0) { ++n; sum += 0.025641 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 1.0,  2.0)); if (c.a != 0.0) { ++n; sum += 0.014652 * c; }",
		"	c = texture2DRect(texture, coord+vec2( 2.0,  2.0)); if (c.a != 0.0) { ++n; sum += 0.003663 * c; }",
		"	c = texture2DRect(texture, coord                 ); if (c.a != 0.0) { ++n; sum += 0.150183 * c; }",
		"",
		"	gl_FragColor = (n > 0 && n < 25) ? sum : c;",
		"}"
	};

	private static String[] createChromaKeySource(boolean maskOnly) {
		return new String[] {
				"uniform sampler2DRect texture;",
				"uniform vec2 keyUV;",
				"uniform float similarity;",
				"uniform float blend;",
				"uniform float t_minus_c;",
				"uniform float cutoff;",
				"",
				"const vec3 yVec = vec3(0.299, 0.587, 0.114);",
				"const vec3 uVec = vec3(-0.14713, -0.28886, 0.436);",
				"const vec3 vVec = vec3(0.615, -0.51499, -0.10001);",
				"",
				"void main(void)",
				"{",
				"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
				"	vec3 rgb = color.rgb/color.a;",
				"",
				"	vec2 uv = vec2(dot(uVec, rgb), dot(vVec, rgb));",
				"	float d = distance(keyUV, uv);",
				"	float a1 = (d-similarity)/blend;",
				"",
				"	float y = dot(yVec, rgb);",
				"	float a2 = clamp(1.0-(y-cutoff)/t_minus_c, 0.0, 1.0);",
				"",
				"	if (a1 > 0.0) {",
				"		float a = min(1.0, a1)*(1.0-a2)+a2;",
	 maskOnly ? "		gl_FragData[0] = vec4(a, a, a, 1.0);"
			  : "		gl_FragData[0] = color*a; gl_FragData[1] = vec4(a);",
				"	} else {",
	 maskOnly ? "		gl_FragData[0] = (a2 == 1.0) ? vec4(1.0) : vec4(0.0, 0.0, 0.0, 1.0);"
			  : "		gl_FragData[0] = (a2 == 1.0) ? color : vec4(0.0, 0.0, 0.0, color.a*a2); gl_FragData[1] = vec4(0.0);",
				"	}",
				"}"
		};
	}

	@ShaderSource
	public static final String[] CHROMA_KEY = createChromaKeySource(false);

	@ShaderSource
	public static final String[] CHROMA_KEY_MASK_ONLY = createChromaKeySource(true);

	@ShaderSource
	public static final String[] CHROMA_KEY_SMOOTHING_LOW = {
		"uniform sampler2DRect tex0;",
		"uniform sampler2DRect tex1;",
		"",
		"void main(void)",
		"{",
		"	vec2 coord = gl_TexCoord[0].st;",
		"	vec4 sum = vec4(0.0);",
		"	int n = 0;",
		"	vec2 d;",
		"	vec4 c;",
		"	d = coord+vec2(-1.0, -1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.0625 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 0.0, -1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.1250 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 1.0, -1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.0625 * texture2DRect(tex0, d);",
		"	d = coord+vec2(-1.0,  0.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.1250 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 1.0,  0.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.1250 * texture2DRect(tex0, d);",
		"	d = coord+vec2(-1.0,  1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.0625 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 0.0,  1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.1250 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 1.0,  1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.0625 * texture2DRect(tex0, d);",
		"	d = coord                 ; c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; c = texture2DRect(tex0, d); sum += 0.2500 * c;",
		"",
		"	gl_FragColor = (n > 0 && n < 9) ? sum : c;",
		"}"
	};

	@ShaderSource
	public static final String[] CHROMA_KEY_SMOOTHING_HIGH = {
		"uniform sampler2DRect tex0;",
		"uniform sampler2DRect tex1;",
		"",
		"void main(void)",
		"{",
		"	vec2 coord = gl_TexCoord[0].st;",
		"	vec4 sum = vec4(0.0);",
		"	int n = 0;",
		"	vec2 d;",
		"	vec4 c;",
		"	d = coord+vec2(-2.0, -2.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.003663 * texture2DRect(tex0, d);",
		"	d = coord+vec2(-1.0, -2.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.014652 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 0.0, -2.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.025641 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 1.0, -2.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.014652 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 2.0, -2.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.003663 * texture2DRect(tex0, d);",
		"	d = coord+vec2(-2.0, -1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.014652 * texture2DRect(tex0, d);",
		"	d = coord+vec2(-1.0, -1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.058608 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 0.0, -1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.095238 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 1.0, -1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.058608 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 2.0, -1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.014652 * texture2DRect(tex0, d);",
		"	d = coord+vec2(-2.0,  0.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.025641 * texture2DRect(tex0, d);",
		"	d = coord+vec2(-1.0,  0.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.095238 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 1.0,  0.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.095238 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 2.0,  0.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.025641 * texture2DRect(tex0, d);",
		"	d = coord+vec2(-2.0,  1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.014652 * texture2DRect(tex0, d);",
		"	d = coord+vec2(-1.0,  1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.058608 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 0.0,  1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.095238 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 1.0,  1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.058608 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 2.0,  1.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.014652 * texture2DRect(tex0, d);",
		"	d = coord+vec2(-2.0,  2.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.003663 * texture2DRect(tex0, d);",
		"	d = coord+vec2(-1.0,  2.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.014652 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 0.0,  2.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.025641 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 1.0,  2.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.014652 * texture2DRect(tex0, d);",
		"	d = coord+vec2( 2.0,  2.0); c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; sum += 0.003663 * texture2DRect(tex0, d);",
		"	d = coord                 ; c = texture2DRect(tex1, d); if (c.a != 0.0) { ++n; }; c = texture2DRect(tex0, d); sum += 0.150183 * c;",
		"",
		"	gl_FragColor = (n > 0 && n < 25) ? sum : c;",
		"}"
	};

	@ShaderSource(program=false)
	public static final String[] rgb2hsl = {
		"vec3 rgb2hsl(vec3 rgb) {",
		"	float min = min(min(rgb.r, rgb.g), rgb.b);",
		"	float max = max(max(rgb.r, rgb.g), rgb.b);",
		"	float dmax = max - min;",
		"",
		"	float luma = (max + min)*0.5;",
		"	float hue, sat;",
		"",
		"	if (dmax == 0.0) {",
		"		hue = sat = 0.0;",
		"	} else {",
		"		sat = (luma < 0.5) ? dmax/(max+min) : dmax/(2.0-max-min);",
		"",
		"		float dr = ((max-rgb.r)/6.0 + dmax/2.0)/dmax;",
		"		float dg = ((max-rgb.g)/6.0 + dmax/2.0)/dmax;",
		"		float db = ((max-rgb.b)/6.0 + dmax/2.0)/dmax;",
		"",
		"		hue = (rgb.r == max) ? db-dg",
		"			: (rgb.g == max) ? 1.0/3.0 + dr-db",
		"			:				   2.0/3.0 + dg-dr;",
		"",
		"		if (hue < 0.0) hue += 1.0;",
		"		else if (hue > 1.0) hue -= 1.0;",
		"	}",
		"",
		"	return vec3(hue, sat, luma);",
		"}",
	};

	private static String[] createHSLKeySource(boolean maskOnly) {
		List<String> list = new ArrayList<String>(Arrays.asList(new String[]{
				"vec3 rgb2hsl(vec3 rgb);",
				"",
				"uniform sampler2DRect texture;",
				"uniform vec3 similarityMin;",
				"uniform vec3 similarityMax;",
				"uniform vec3 blendMin;",
				"uniform vec3 blendMax;",
				"uniform vec3 bleachHSL;",
				"uniform vec3 bleachRGB;",
				"uniform vec2 correctionHue;",
				"uniform float correctionSat;",
				"uniform float correctionAlpha;",
				"",
				"const vec3 vec3_111 = vec3(1.0);",
				"const float PI = 3.14159265358979323846264;",
				"",
				"void main(void)",
				"{",
				"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
				"	vec3 unmult = color.rgb/color.a;",
				"	vec3 hsl = rgb2hsl(unmult);",
				"	vec3 bleachDelta = hsl-bleachHSL;",
				"	hsl.y *= 1.0-abs(hsl.z-0.5)*2.0;",
				"",
				"	vec3 a3 = max((similarityMin-hsl)/blendMin, (hsl-similarityMax)/blendMax);",
				"	a3.x = min(a3.x, (similarityMin.x-(hsl.x-1.0))/blendMin.x);",
				"	a3.x = min(a3.x, ((hsl.x+1.0)-similarityMax.x)/blendMax.x);",
				"",
				"	a3 = vec3_111 - clamp(a3, 0.0, 1.0);",
				"	a3 = 0.5*(1.0-cos(PI*a3));",
				"	float a = 1.0 - a3.x*a3.y*a3.z;",
				"",
				"	if (similarityMin.y-blendMin.y < 0.0 && hsl.y < blendMin.y-similarityMin.y) {",
				"		float y = 1.0 - clamp((hsl.y+similarityMin.y)/blendMin.y, 0.0, 1.0);",
				"		y = 0.5*(1.0-cos(PI*y));",
				"		a = min(a, 1.0 - y*a3.z);",
				"	}",
				"",
				"	float bleachLevel = (bleachDelta.x-floor(bleachDelta.x))*6.0;",
				"	if (bleachLevel > 3.0) {",
				"		bleachLevel = (6.0-bleachLevel)/correctionHue.x;",
				"	} else {",
				"		bleachLevel /= correctionHue.y;",
				"	}",
				"",
				"	vec3 b3 = vec3_111 - vec3(min(bleachLevel, 1.0), min(abs(bleachDelta.y)*correctionSat, 1.0), min(abs(bleachDelta.z)*2.0, 1.0));",
				"	bleachLevel = b3.x*b3.y*b3.z;",
				"	bleachLevel = max(bleachLevel, (abs(bleachDelta.y)*correctionSat+(1.0-correctionSat*2.0))*b3.z);",
				"",
				"	a = (1.0-correctionAlpha*bleachLevel) * a * color.a;",
			}));

		if (maskOnly) {
			list.addAll(Arrays.asList(new String[] {
					"	gl_FragColor = vec4(a, a, a, 1.0);",
					"}"
			}));
		} else {
			list.addAll(Arrays.asList(new String[] {
					"	unmult = clamp((unmult-bleachLevel*bleachRGB)/(1.0-bleachLevel), 0.0, 1.0);",
					"	gl_FragColor = vec4(unmult, 1.0) * a;",
					"}"
			}));
		}

		return list.toArray(new String[list.size()]);
	};

	@ShaderSource(attach={"rgb2hsl"})
	public static final String[] HSL_KEY = createHSLKeySource(false);

	@ShaderSource(attach={"rgb2hsl"})
	public static final String[] HSL_KEY_MASK_ONLY = createHSLKeySource(true);

}
