/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.stylize;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.BlendMode;
import ch.kuramo.javie.effects.BlendModeShaders;
import ch.kuramo.javie.effects.VideoEffectUtil;
import ch.kuramo.javie.effects.blurSharpen.BlurUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Glow", category=Effect.STYLIZE)
public class Glow {

	public enum GlowColors { ORIGINAL_COLORS, A_AND_B_COLORS, A_AND_B_COLORS_ALPHA_BASE }

	public enum GlowComposite { ON_TOP, BEHIND, NONE }

	public enum GlowDimensions { BOTH, HORIZONTAL, VERTICAL }

	public enum GlowLooping { SAWTOOTH_AB, SAWTOOTH_BA, TRIANGLE_ABA, TRIANGLE_BAB }


	@Property("ORIGINAL_COLORS")
	private IAnimatableEnum<GlowColors> colors;

	@Property(value="60", min="0", max="100")
	private IAnimatableDouble threshold;

	@Property(value="10", min="0", max="500")
	private IAnimatableDouble radius;

	@Property(value="1", min="0", max="255")
	private IAnimatableDouble intensity;

	@Property("NONE")
	private IAnimatableEnum<GlowComposite> composite;

	@Property("ADD")
	private IAnimatableEnum<BlendMode> operation;

	@Property("TRIANGLE_ABA")
	private IAnimatableEnum<GlowLooping> colorLooping;

	@Property(value="1", min="1", max="127")
	private IAnimatableDouble colorLoops;

	@Property
	private IAnimatableDouble colorPhase;

	@Property(value="50", min="0", max="100")
	private IAnimatableDouble abMidpoint;

	@Property("1,1,1")
	private IAnimatableColor colorA;

	@Property("0,0,0")
	private IAnimatableColor colorB;

	@Property
	private IAnimatableEnum<GlowDimensions> dimensions;

	@Property("true")
	private IAnimatableBoolean fast;


	private final IVideoEffectContext context;

	private final IShaderRegistry shaders;

	private final IShaderProgram oColorsThresholdProgram;

	private final IShaderProgram abColorsThresholdProgram;

	private final IShaderProgram alphaBaseThresholdProgram;

	private final IShaderProgram sawtoothProgram;

	private final IShaderProgram triangleProgram;

	private final IShaderProgram oColorsIntensityProgram;

	private final IShaderProgram oColorsIntensityFloatProgram;

	private final IShaderProgram abColorsIntensityProgram;

	@Inject
	public Glow(IVideoEffectContext context, IShaderRegistry shaders) {
		this.context = context;
		this.shaders = shaders;
		oColorsThresholdProgram = shaders.getProgram(Glow.class, "OCOLORS_THRESHOLD");
		abColorsThresholdProgram = shaders.getProgram(Glow.class, "ABCOLORS_THRESHOLD");
		alphaBaseThresholdProgram = shaders.getProgram(Glow.class, "ALPHA_BASE_THRESHOLD");
		sawtoothProgram = shaders.getProgram(Glow.class, "SAWTOOTH");
		triangleProgram = shaders.getProgram(Glow.class, "TRIANGLE");
		oColorsIntensityProgram = shaders.getProgram(Glow.class, "OCOLORS_INTENSITY");
		oColorsIntensityFloatProgram = shaders.getProgram(Glow.class, "OCOLORS_INTENSITY_FLOAT");
		abColorsIntensityProgram = shaders.getProgram(Glow.class, "ABCOLORS_INTENSITY");
	}

	public VideoBounds getVideoBounds() {
		// GaussianBlurとほぼ同じ。

		double radius = context.value(this.radius);
		if (radius == 0) {
			return null;
		}

		GlowDimensions dimensions = context.value(this.dimensions);
		boolean horz = (dimensions != GlowDimensions.VERTICAL);
		boolean vert = (dimensions != GlowDimensions.HORIZONTAL);

		// blurが50より大きい場合はfastプロパティの値にかかわらず高速モード
		boolean fast = (radius > 50) || context.value(this.fast);

		// 解像度に合わせてblurの値を変換
		radius = context.getVideoResolution().scale(radius);


		VideoBounds bounds = context.getPreviousBounds();
		if (bounds.isEmpty()) {
			return bounds;
		}

		int sampleRatio = 1;
		if (fast) {
			for (int factor : BlurUtil.getDownSampleFactors(radius)) {
				int hFactor = horz ? factor : 1;
				int vFactor = vert ? factor : 1;
				bounds = new VideoBounds(bounds.x/hFactor, bounds.y/vFactor,
						(bounds.width+hFactor-1)/hFactor, (bounds.height+vFactor-1)/vFactor);
				sampleRatio *= factor;
			}
			radius /= sampleRatio;
		}

		int ceiledRadius = (int) Math.ceil(radius);


		bounds = VideoEffectUtil.expandBounds(bounds, ceiledRadius, horz, vert);

		int hRatio = horz ? sampleRatio : 1;
		int vRatio = vert ? sampleRatio : 1;
		return new VideoBounds(bounds.x*hRatio, bounds.y*vRatio,
				bounds.width*hRatio, bounds.height*vRatio);
	}

	public IVideoBuffer doVideoEffect() {
		GlowColors colors = context.value(this.colors);

		IVideoBuffer input = null;
		IVideoBuffer buf1 = null;
		IVideoBuffer buf2 = null;
		IVideoBuffer buf3 = null;
		IVideoBuffer buf4 = null;
		IVideoBuffer buf5 = null;
		try {
			input = context.doPreviousEffect();
			if (input.getBounds().isEmpty()) {
				IVideoBuffer result = input;
				input = null;
				return result;
			}

			buf1 = threshold(input, colors);

			buf2 = blur(buf1);
			if (buf2 == buf1) buf1 = null;

			if (colors == GlowColors.ORIGINAL_COLORS) {
				buf3 = buf2;
				buf2 = null;
			} else {
				buf3 = abColors(buf2);
			}

			buf4 = intensity(buf3, colors);
			if (buf4 == buf3) buf3 = null;

			buf5 = composite(buf4, input);
			if (buf5 == buf4) buf4 = null;

			IVideoBuffer result = buf5;
			buf5 = null;
			return result;

		} finally {
			if (input != null) input.dispose();
			if (buf1 != null) buf1.dispose();
			if (buf2 != null) buf2.dispose();
			if (buf3 != null) buf3.dispose();
			if (buf4 != null) buf4.dispose();
			if (buf5 != null) buf5.dispose();
		}
	}

	@ShaderSource
	public static final String[] OCOLORS_THRESHOLD = {
		"uniform sampler2DRect texture;",
		"uniform float upper;",
		"uniform float lower;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
		"	float a = color.a;",
		"	vec3 rgb = (a != 0.0) ? color.rgb/a : vec3(0.0);",
		"",
		"	rgb = clamp((rgb - lower) / (upper - lower), 0.0, 1.0);",
		"",
		"	gl_FragColor = vec4(rgb*a, a);",
		"}"
	};

	@ShaderSource
	public static final String[] ABCOLORS_THRESHOLD = {
		"uniform sampler2DRect texture;",
		"uniform float upper;",
		"uniform float lower;",
		"",
		"const vec3 lumaVec = vec3(0.299, 0.587, 0.114);",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
		"",
		"	float luma = dot(color.rgb, lumaVec);",
		"	luma = clamp((luma - lower) / (upper - lower), 0.0, 1.0);",
		"",
		"	gl_FragColor = vec4(luma);",
		"}"
	};

	@ShaderSource
	public static final String[] ALPHA_BASE_THRESHOLD = {
		"uniform sampler2DRect texture;",
		"uniform float upper;",
		"uniform float lower;",
		"",
		"void main(void)",
		"{",
		"	float a = texture2DRect(texture, gl_TexCoord[0].st).a;",
		"",
		"	a = clamp((a - lower) / (upper - lower), 0.0, 1.0);",
		"",
		"	gl_FragColor = vec4(a);",
		"}"
	};

	private IVideoBuffer threshold(IVideoBuffer input, GlowColors colors) {
		final IShaderProgram program =
				(colors == GlowColors.ORIGINAL_COLORS) ? oColorsThresholdProgram :
				(colors == GlowColors.A_AND_B_COLORS) ? abColorsThresholdProgram : alphaBaseThresholdProgram;

		double threshold = context.value(this.threshold) / 100;
		final double upper = Math.min(threshold + 0.0625, 1);
		final double lower = Math.max(threshold - 0.0625, 0);

		final GL2 gl = context.getGL().getGL2();

		VideoBounds bounds = input.getBounds();
		final int w = bounds.width;
		final int h = bounds.height;

		VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

		IVideoBuffer buffer = null;
		try {
			buffer = context.createVideoBuffer(bounds);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, buffer.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, input.getTexture());

			program.useProgram(new Runnable() {
				public void run() {
					gl.glUniform1i(program.getUniformLocation("texture"), 0);
					gl.glUniform1f(program.getUniformLocation("upper"), (float)upper);
					gl.glUniform1f(program.getUniformLocation("lower"), (float)lower);

					gl.glBegin(GL2.GL_QUADS);
					gl.glTexCoord2f(0, 0);
					gl.glVertex2f(0, 0);
					gl.glTexCoord2f(w, 0);
					gl.glVertex2f(w, 0);
					gl.glTexCoord2f(w, h);
					gl.glVertex2f(w, h);
					gl.glTexCoord2f(0, h);
					gl.glVertex2f(0, h);
					gl.glEnd();
				}
			});

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	private IVideoBuffer blur(IVideoBuffer input) {
		double radius = context.value(this.radius);
		if (radius == 0) {
			return input;
		}

		GlowDimensions dimensions = context.value(this.dimensions);
		boolean horz = (dimensions != GlowDimensions.VERTICAL);
		boolean vert = (dimensions != GlowDimensions.HORIZONTAL);

		boolean fast = (radius > 50) || context.value(this.fast);

		radius = context.getVideoResolution().scale(radius);

		GL2 gl = context.getGL().getGL2();
		GLU glu = context.getGLU();

		return blur(input, radius, false, horz, vert, fast, gl, glu);
	}

	private IVideoBuffer blur(
			IVideoBuffer input, double blur, boolean repeatEdgePixels,
			boolean horz, boolean vert, boolean fast, GL2 gl, GLU glu) {

		// TODO 以下は GaussianBlur から丸々コピペしたもの。整理する。

		VideoBounds inputBounds = input.getBounds();

		int sampleRatio = 1;
		if (fast) {
			VideoBounds bounds = inputBounds;
			for (int factor : BlurUtil.getDownSampleFactors(blur)) {
				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(input, gl);
				}

				int hFactor = horz ? factor : 1;
				int vFactor = vert ? factor : 1;
				bounds = new VideoBounds(bounds.x/hFactor, bounds.y/vFactor,
						(bounds.width+hFactor-1)/hFactor, (bounds.height+vFactor-1)/vFactor);

				IVideoBuffer buf = context.createVideoBuffer(bounds);
				BlurUtil.doDownSample(input, buf, hFactor, vFactor, gl, glu, shaders);
				input.dispose();
				input = buf;

				sampleRatio *= factor;
			}
			blur /= sampleRatio;
		}

		int radius = (int) Math.ceil(blur);


		// 標準偏差の2.5倍の位置がぼかしの端となるようにする（これでだいたいAEと同じになる）
		double sigma = blur / 2.5;
		double sigmaSquare = sigma * sigma;

		float[] kernel = new float[radius*2+1];
		float sum = 0;
		for (int i = 1; i <= radius; ++i) {
			sum += 2 * (kernel[radius-i] = (float) Math.exp(-i * i / (2 * sigmaSquare)));
		}
		kernel[radius] = 1 / (++sum);
		for (int i = 1; i <= radius; ++i) {
			kernel[radius+i] = (kernel[radius-i] /= sum);
		}


		IVideoBuffer buf1 = null, buf2 = null, buf3 = null;
		try {
			if (horz) {
				VideoBounds bounds = input.getBounds();

				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(input, gl);
				} else {
					bounds = VideoEffectUtil.expandBounds(bounds, radius, true, false);
				}

				buf1 = context.createVideoBuffer(bounds);
				VideoEffectUtil.convolution1D(input, buf1, true, radius, kernel, gl, glu, shaders);
			} else {
				buf1 = input;
				input = null;
			}

			if (vert) {
				VideoBounds bounds = buf1.getBounds();

				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(buf1, gl);
				} else {
					bounds = VideoEffectUtil.expandBounds(bounds, radius, false, true);
				}

				buf2 = context.createVideoBuffer(bounds);
				VideoEffectUtil.convolution1D(buf1, buf2, false, radius, kernel, gl, glu, shaders);
			} else {
				buf2 = buf1;
				buf1 = null;
			}

			if (sampleRatio != 1) {
				int hRatio = horz ? sampleRatio : 1;
				int vRatio = vert ? sampleRatio : 1;
				VideoBounds bounds;

				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(buf2, gl);
					bounds = inputBounds;
				} else {
					bounds = buf2.getBounds();
					bounds = new VideoBounds(bounds.x*hRatio, bounds.y*vRatio, bounds.width*hRatio, bounds.height*vRatio);
				}

				buf3 = context.createVideoBuffer(bounds);
				BlurUtil.doUpSample(buf2, buf3, hRatio, vRatio, gl, glu);
			} else {
				buf3 = buf2;
				buf2 = null;
			}

			return buf3;

		} finally {
			if (input != null) {
				input.dispose();
			}
			if (buf1 != null) {
				buf1.dispose();
			}
			if (buf2 != null) {
				buf2.dispose();
			}
		}
	}

	@ShaderSource
	public static final String[] SAWTOOTH = {
		"uniform sampler2DRect texture;",
		"uniform float loops;",
		"uniform float phase;",
		"uniform float midpoint;",
		"uniform vec3 colorA;",
		"uniform vec3 colorM;",
		"uniform vec3 colorB;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
		"	float a = color.a;",
		"	float t = phase + a * loops;",
		"",
		"	if (t > 1.0) {",
		"		t -= floor(t);",
		"	}",
		"",
		"	vec3 rgb;",
		"	if (t > midpoint) {",
		"		t = (t-midpoint)/(1.0-midpoint);",
		"		rgb = colorA*t + colorM*(1.0-t);",
		"	} else {",
		"		t = t/midpoint;",
		"		rgb = colorB*(1.0-t) + colorM*t;",
		"	}",
		"",
		"	gl_FragColor = vec4(rgb, 1.0)*a;",
		"}"
	};

	@ShaderSource
	public static final String[] TRIANGLE = {
		"uniform sampler2DRect texture;",
		"uniform float loops;",
		"uniform float phase;",
		"uniform float midpoint;",
		"uniform vec3 colorA;",
		"uniform vec3 colorM;",
		"uniform vec3 colorB;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
		"	float a = color.a;",
		"	float t = phase + a * loops;",
		"",
		"	t = 1.0 - 2.0 * abs(t - floor(t) - 0.5);",	// SAWTOOTHとはこの１行が異なるだけ。
		"",
		"	vec3 rgb;",
		"	if (t > midpoint) {",
		"		t = (t-midpoint)/(1.0-midpoint);",
		"		rgb = colorA*t + colorM*(1.0-t);",
		"	} else {",
		"		t = t/midpoint;",
		"		rgb = colorB*(1.0-t) + colorM*t;",
		"	}",
		"",
		"	gl_FragColor = vec4(rgb, 1.0)*a;",
		"}"
	};

	private IVideoBuffer abColors(IVideoBuffer input) {
		GlowLooping looping = context.value(colorLooping);
		final double loops = context.value(colorLoops);
		final double phase = context.value(colorPhase) / 360;
		final double midpoint = 1 - context.value(abMidpoint) / 100;

		final Color colorA;
		final Color colorB;
		if (looping == GlowLooping.SAWTOOTH_AB || looping == GlowLooping.TRIANGLE_ABA) {
			colorA = context.value(this.colorA);
			colorB = context.value(this.colorB);
		} else {
			colorA = context.value(this.colorB);
			colorB = context.value(this.colorA);
		}
		final Color colorM = new Color((colorA.r+colorB.r)/2, (colorA.g+colorB.g)/2, (colorA.b+colorB.b)/2);

		final IShaderProgram program = (looping == GlowLooping.SAWTOOTH_AB || looping == GlowLooping.SAWTOOTH_BA)
				? sawtoothProgram : triangleProgram;

		final GL2 gl = context.getGL().getGL2();

		VideoBounds bounds = input.getBounds();
		final int w = bounds.width;
		final int h = bounds.height;

		VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

		IVideoBuffer buffer = null;
		try {
			buffer = context.createVideoBuffer(bounds);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, buffer.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, input.getTexture());

			program.useProgram(new Runnable() {
				public void run() {
					gl.glUniform1i(program.getUniformLocation("texture"), 0);
					gl.glUniform1f(program.getUniformLocation("loops"), (float)loops);
					gl.glUniform1f(program.getUniformLocation("phase"), (float)phase);
					gl.glUniform1f(program.getUniformLocation("midpoint"), (float)midpoint);
					gl.glUniform3f(program.getUniformLocation("colorA"), (float)colorA.r, (float)colorA.g, (float)colorA.b);
					gl.glUniform3f(program.getUniformLocation("colorM"), (float)colorM.r, (float)colorM.g, (float)colorM.b);
					gl.glUniform3f(program.getUniformLocation("colorB"), (float)colorB.r, (float)colorB.g, (float)colorB.b);

					gl.glBegin(GL2.GL_QUADS);
					gl.glTexCoord2f(0, 0);
					gl.glVertex2f(0, 0);
					gl.glTexCoord2f(w, 0);
					gl.glVertex2f(w, 0);
					gl.glTexCoord2f(w, h);
					gl.glVertex2f(w, h);
					gl.glTexCoord2f(0, h);
					gl.glVertex2f(0, h);
					gl.glEnd();
				}
			});

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	@ShaderSource
	public static final String[] OCOLORS_INTENSITY = {
		"uniform sampler2DRect texture;",
		"uniform float intensity;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
		"	gl_FragColor = vec4(min(color.rgb*intensity, color.a), color.a);",
		"}"
	};

	@ShaderSource
	public static final String[] OCOLORS_INTENSITY_FLOAT = {
		"uniform sampler2DRect texture;",
		"uniform float intensity;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
		"	gl_FragColor = vec4(color.rgb*intensity, color.a);",
		"}"
	};

	@ShaderSource
	public static final String[] ABCOLORS_INTENSITY = {
		"uniform sampler2DRect texture;",
		"uniform float intensity;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2DRect(texture, gl_TexCoord[0].st);",
		"	gl_FragColor = vec4(color.rgb*intensity, min(color.a*intensity, 1.0));",
		"}"
	};

	private IVideoBuffer intensity(IVideoBuffer input, GlowColors colors) {
		final double intensity = context.value(this.intensity);
		if (intensity == 1) {
			return input;
		}

		// A_AND_B_COLORS と A_AND_B_COLORS_ALPHA_BASE のシェーダは同じ
		final IShaderProgram program
				= (colors != GlowColors.ORIGINAL_COLORS) ? abColorsIntensityProgram
				: input.getColorMode().isFloat() ? oColorsIntensityFloatProgram : oColorsIntensityProgram;

		final GL2 gl = context.getGL().getGL2();

		VideoBounds bounds = input.getBounds();
		final int w = bounds.width;
		final int h = bounds.height;

		VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

		IVideoBuffer buffer = null;
		try {
			buffer = context.createVideoBuffer(bounds);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, buffer.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, input.getTexture());

			program.useProgram(new Runnable() {
				public void run() {
					gl.glUniform1i(program.getUniformLocation("texture"), 0);
					gl.glUniform1f(program.getUniformLocation("intensity"), (float)intensity);

					gl.glBegin(GL2.GL_QUADS);
					gl.glTexCoord2f(0, 0);
					gl.glVertex2f(0, 0);
					gl.glTexCoord2f(w, 0);
					gl.glVertex2f(w, 0);
					gl.glTexCoord2f(w, h);
					gl.glVertex2f(w, h);
					gl.glTexCoord2f(0, h);
					gl.glVertex2f(0, h);
					gl.glEnd();
				}
			});

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	private IVideoBuffer composite(IVideoBuffer glow, IVideoBuffer orig) {
		final GlowComposite composite = context.value(this.composite);
		BlendMode operation = context.value(this.operation);

		if (composite == GlowComposite.NONE && operation == BlendMode.NONE) {
			return glow;
		}

		final GL2 gl = context.getGL().getGL2();

		//「エッジピクセルを繰り返す」機能を付ける場合、
		// origはsetClampToEdgeされているかもしれないのでsetClampToBorderしておく必要がある。
		//VideoEffectUtil.setClampToBorder(orig, gl);

		VideoBounds glowBounds = glow.getBounds();
		VideoBounds origBounds = orig.getBounds();
		final int w = glowBounds.width;
		final int h = glowBounds.height;
		final float ox = (float)(glowBounds.x-origBounds.x);
		final float oy = (float)(glowBounds.y-origBounds.y);

		VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

		IVideoBuffer buffer = null;
		try {
			buffer = context.createVideoBuffer(glowBounds);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, buffer.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, glow.getTexture());
			gl.glActiveTexture(GL2.GL_TEXTURE1);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, orig.getTexture());

			final IShaderProgram program = shaders.getProgram(Glow.class, operation.name());
			program.useProgram(new Runnable() {
				public void run() {
					gl.glUniform1i(program.getUniformLocation("texGlow"), 0);
					gl.glUniform1i(program.getUniformLocation("texOrig"), 1);
					gl.glUniform1i(program.getUniformLocation("composite"), composite.ordinal());

					gl.glBegin(GL2.GL_QUADS);
					gl.glMultiTexCoord2f(GL2.GL_TEXTURE0, 0, 0);
					gl.glMultiTexCoord2f(GL2.GL_TEXTURE1, ox, oy);
					gl.glVertex2f(0, 0);
					gl.glMultiTexCoord2f(GL2.GL_TEXTURE0, w, 0);
					gl.glMultiTexCoord2f(GL2.GL_TEXTURE1, ox+w, oy);
					gl.glVertex2f(w, 0);
					gl.glMultiTexCoord2f(GL2.GL_TEXTURE0, w, h);
					gl.glMultiTexCoord2f(GL2.GL_TEXTURE1, ox+w, oy+h);
					gl.glVertex2f(w, h);
					gl.glMultiTexCoord2f(GL2.GL_TEXTURE0, 0, h);
					gl.glMultiTexCoord2f(GL2.GL_TEXTURE1, ox, oy+h);
					gl.glVertex2f(0, h);
					gl.glEnd();
				}
			});

			gl.glActiveTexture(GL2.GL_TEXTURE1);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	private static final String[] createCompositeSource(String name) {
		boolean none = name.equals("none");
		boolean noneOrNormal = none || name.equals("normal");

		return new String[] {
								"uniform sampler2DRect texGlow;",
								"uniform sampler2DRect texOrig;",
								"uniform int composite;",
								"",
								"vec4 blend_normal(vec4 pDst, vec4 pSrc, float intensity);",
				 noneOrNormal ? ""
				: String.format("vec4 blend_%s(vec4 pDst, vec4 pSrc, float intensity);", name),
								"",
								"void main(void)",
								"{",
								"	vec4 glow = texture2DRect(texGlow, gl_TexCoord[0].st);",
								"	vec4 orig = texture2DRect(texOrig, gl_TexCoord[1].st);",
						none ? ""
				: String.format("	glow = blend_%s(glow, orig, 1.0);", name),
								"",
								"	gl_FragColor = (composite == 0) ? blend_normal(glow, orig, 1.0) :",
								"				   (composite == 1) ? blend_normal(orig, glow, 1.0) : glow;",
								"}"
		};
	}

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] NONE = createCompositeSource("none");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] NORMAL = createCompositeSource("normal");

//	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
//	public static final String[] DISSOLVE = createCompositeSource("dissolve");
//
//	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
//	public static final String[] DANCING_DISSOLVE = createCompositeSource("dancing_dissolve");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] DARKEN = createCompositeSource("darken");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] MULTIPLY = createCompositeSource("multiply");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] COLOR_BURN = createCompositeSource("color_burn");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] LINEAR_BURN = createCompositeSource("linear_burn");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] DARKER_COLOR = createCompositeSource("darker_color");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] ADD = createCompositeSource("add");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] LIGHTEN = createCompositeSource("lighten");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] SCREEN = createCompositeSource("screen");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] COLOR_DODGE = createCompositeSource("color_dodge");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] LINEAR_DODGE = createCompositeSource("linear_dodge");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] LIGHTER_COLOR = createCompositeSource("lighter_color");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] OVERLAY = createCompositeSource("overlay");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] SOFT_LIGHT = createCompositeSource("soft_light");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] HARD_LIGHT = createCompositeSource("hard_light");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] LINEAR_LIGHT = createCompositeSource("linear_light");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] VIVID_LIGHT = createCompositeSource("vivid_light");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] PIN_LIGHT = createCompositeSource("pin_light");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] HARD_MIX = createCompositeSource("hard_mix");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] DIFFERENCE = createCompositeSource("difference");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] EXCLUSION = createCompositeSource("exclusion");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] HUE = createCompositeSource("hue");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] SATURATION = createCompositeSource("saturation");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] COLOR = createCompositeSource("color");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] LUMINOSITY = createCompositeSource("luminosity");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] STENCIL_ALPHA = createCompositeSource("stencil_alpha");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] STENCIL_LUMA = createCompositeSource("stencil_luma");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] SILHOUETTE_ALPHA = createCompositeSource("silhouette_alpha");

	@ShaderSource(attach=BlendModeShaders.BLEND_FUNCTIONS)
	public static final String[] SILHOUETTE_LUMA = createCompositeSource("silhouette_luma");

}
