/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.transition;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.GradientWipe", category=Effect.TRANSITION)
public class GradientWipe {

	@ShaderSource
	public static final String[] GRADIENT_WIPE_NORMAL = createShaderSource(false, false);

	@ShaderSource
	public static final String[] GRADIENT_WIPE_NORMAL_TILE = createShaderSource(false, true);

	@ShaderSource
	public static final String[] GRADIENT_WIPE_INVERT = createShaderSource(true, false);

	@ShaderSource
	public static final String[] GRADIENT_WIPE_INVERT_TILE = createShaderSource(true, true);


	private static String[] createShaderSource(boolean invert, boolean tile) {
		return new String[] {
				"uniform sampler2DRect grdTex;",
				"uniform sampler2DRect srcTex;",
				"uniform float min;",
				"uniform float max;",

		 tile ? "uniform vec2 size;"
			  : "",

				"",
				"const vec3 yvec = vec3(0.299, 0.587, 0.114);",
				"",
				"void main(void)",
				"{",
				"	vec2 texCoord = gl_TexCoord[0].st;",

		 tile ? "	vec4 grd = texture2DRect(grdTex, texCoord - floor(texCoord/size)*size);"
			  : "	vec4 grd = texture2DRect(grdTex, texCoord);",

				"	vec4 src = texture2DRect(srcTex, gl_FragCoord.xy);",

	   invert ? "	float y = 1.0 - dot(grd.rgb, yvec);"
			  : "	float y = dot(grd.rgb, yvec);",

				"	float a = clamp((y-min)/(max-min), 0.0, 1.0);",
				"	gl_FragColor = src*a;",
				"}"
		};
	}


	public enum GradientPlacement { TILE, CENTER, STRETCH }


	private final IVideoEffectContext context;

	private final IShaderProgram normalProgram;

	private final IShaderProgram normalTileProgram;

	private final IShaderProgram invertProgram;

	private final IShaderProgram invertTileProgram;

	@Property(min="0", max="100")
	private IAnimatableDouble transitionCompletion;

	@Property(min="0", max="100")
	private IAnimatableDouble transitionSoftness;

	@Property
	private IAnimatableLayerReference gradientLayer;

	@Property("STRETCH")
	private IAnimatableEnum<GradientPlacement> gradientPlacement;

	@Property
	private IAnimatableBoolean invertGradient;


	@Inject
	public GradientWipe(IVideoEffectContext context, IShaderRegistry shaders) {
		this.context = context;
		this.normalProgram = shaders.getProgram(GradientWipe.class, "GRADIENT_WIPE_NORMAL");
		this.normalTileProgram = shaders.getProgram(GradientWipe.class, "GRADIENT_WIPE_NORMAL_TILE");
		this.invertProgram = shaders.getProgram(GradientWipe.class, "GRADIENT_WIPE_INVERT");
		this.invertTileProgram = shaders.getProgram(GradientWipe.class, "GRADIENT_WIPE_INVERT_TILE");
	}

	public IVideoBuffer doVideoEffect() {
		double completion = context.value(transitionCompletion) / 100;
		if (completion == 0) {
			return null;
		} else if (completion == 1) {
			IVideoBuffer vb = context.createVideoBuffer(context.getPreviousBounds());
			VideoEffectUtil.clearTexture(vb, context.getGL().getGL2());
			return vb;
		}

		double softness = context.value(transitionSoftness) / 100;
		final GradientPlacement placement = context.value(gradientPlacement);
		final boolean tile = (placement == GradientPlacement.TILE);
		boolean invert = context.value(invertGradient);

		final double max = (1 + softness) * completion;
		final double min = max - softness;

		IVideoBuffer sourceBuffer = null;
		IVideoBuffer gradientBuffer = null;
		IVideoBuffer resultBuffer = null;
		try {
			sourceBuffer = context.doPreviousEffect();
			final VideoBounds sourceBounds = sourceBuffer.getBounds();
			if (sourceBounds.isEmpty()) {
				IVideoBuffer result = sourceBuffer;
				sourceBuffer = null;
				return result;
			}

			final GL2 gl = context.getGL().getGL2();

			gradientBuffer = context.getLayerVideoFrame(gradientLayer);
			if (gradientBuffer == null) {
				gradientBuffer = context.createVideoBuffer(new VideoBounds(0, 0));
				VideoEffectUtil.clearTexture(gradientBuffer, gl);
			}
			final VideoBounds gradientBounds = gradientBuffer.getBounds();

			resultBuffer = context.createVideoBuffer(sourceBounds);


			VideoEffectUtil.ortho2D(gl, context.getGLU(), sourceBounds.width, sourceBounds.height);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, resultBuffer.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, gradientBuffer.getTexture());
			gl.glActiveTexture(GL2.GL_TEXTURE1);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, sourceBuffer.getTexture());

			final IShaderProgram program = tile ? (invert ? invertTileProgram : normalTileProgram)
												: (invert ? invertProgram : normalProgram);
			program.useProgram(new Runnable() {
				public void run() {
					gl.glUniform1i(program.getUniformLocation("grdTex"), 0);
					gl.glUniform1i(program.getUniformLocation("srcTex"), 1);
					gl.glUniform1f(program.getUniformLocation("min"), (float)min);
					gl.glUniform1f(program.getUniformLocation("max"), (float)max);

					if (tile) {
						gl.glUniform2f(program.getUniformLocation("size"), gradientBounds.width, gradientBounds.height);
					}

					gl.glBegin(GL2.GL_QUADS);

					switch (placement) {
						case CENTER: {
							float s = (gradientBounds.width - sourceBounds.width) / 2f;
							float t = (gradientBounds.height - sourceBounds.height) / 2f;
							gl.glTexCoord2f(s, t);
							gl.glVertex2f(0, 0);
							gl.glTexCoord2f(s + sourceBounds.width, t);
							gl.glVertex2f(sourceBounds.width, 0);
							gl.glTexCoord2f(s + sourceBounds.width, t + sourceBounds.height);
							gl.glVertex2f(sourceBounds.width, sourceBounds.height);
							gl.glTexCoord2f(s, t + sourceBounds.height);
							gl.glVertex2f(0, sourceBounds.height);
							break;
						}

						case STRETCH:
							gl.glTexCoord2f(0, 0);
							gl.glVertex2f(0, 0);
							gl.glTexCoord2f(gradientBounds.width, 0);
							gl.glVertex2f(sourceBounds.width, 0);
							gl.glTexCoord2f(gradientBounds.width, gradientBounds.height);
							gl.glVertex2f(sourceBounds.width, sourceBounds.height);
							gl.glTexCoord2f(0, gradientBounds.height);
							gl.glVertex2f(0, sourceBounds.height);
							break;

						default:
							gl.glTexCoord2f(0, 0);
							gl.glVertex2f(0, 0);
							gl.glTexCoord2f(sourceBounds.width, 0);
							gl.glVertex2f(sourceBounds.width, 0);
							gl.glTexCoord2f(sourceBounds.width, sourceBounds.height);
							gl.glVertex2f(sourceBounds.width, sourceBounds.height);
							gl.glTexCoord2f(0, sourceBounds.height);
							gl.glVertex2f(0, sourceBounds.height);
							break;
					}

					gl.glEnd();
				}
			});

			gl.glActiveTexture(GL2.GL_TEXTURE1);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

			IVideoBuffer output = resultBuffer;
			resultBuffer = null;
			return output;

		} finally {
			if (sourceBuffer != null) sourceBuffer.dispose();
			if (gradientBuffer != null) gradientBuffer.dispose();
			if (resultBuffer != null) resultBuffer.dispose();
		}
	}

}
