/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.utility;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.ChangeBounds", category=Effect.UTILITY)
public class ChangeBounds {

	private final IVideoEffectContext context;

	@Property(max="1000")	// TODO 最大値はどのくらいまで許容すればよいだろうか。以下同様。
	private IAnimatableInteger	left;

	@Property(max="1000")
	private IAnimatableInteger	top;

	@Property(max="1000")
	private IAnimatableInteger	right;

	@Property(max="1000")
	private IAnimatableInteger	bottom;


	@Inject
	public ChangeBounds(IVideoEffectContext context) {
		this.context = context;
	}

	public VideoBounds getVideoBounds() {
		Resolution resolution = context.getVideoResolution();
		int left = (int) resolution.scale(context.value(this.left));
		int top = (int) resolution.scale(context.value(this.top));
		int right = (int) resolution.scale(context.value(this.right));
		int bottom = (int) resolution.scale(context.value(this.bottom));

		VideoBounds bounds = context.getPreviousBounds();

		return new VideoBounds(
				bounds.x - left,
				bounds.y - top,
				Math.max(0, bounds.width + left + right),
				Math.max(0, bounds.height + top + bottom));
	}

	public IVideoBuffer doVideoEffect() {
		Resolution resolution = context.getVideoResolution();
		int left = (int) resolution.scale(context.value(this.left));
		int top = (int) resolution.scale(context.value(this.top));
		int right = (int) resolution.scale(context.value(this.right));
		int bottom = (int) resolution.scale(context.value(this.bottom));

		IVideoBuffer input = context.doPreviousEffect();
		VideoBounds oldBounds = input.getBounds();

		VideoBounds newBounds = new VideoBounds(
				oldBounds.x - left,
				oldBounds.y - top,
				Math.max(0, oldBounds.width + left + right),
				Math.max(0, oldBounds.height + top + bottom));

		if (newBounds.equals(oldBounds)) {
			return input;
		}

		if (newBounds.isEmpty()) {
			input.dispose();
			return context.createVideoBuffer(newBounds);
		}


		IVideoBuffer buffer = context.createVideoBuffer(newBounds);
		try {
			GL2 gl = context.getGL().getGL2();
			GLU glu = context.getGLU();

			VideoEffectUtil.clearTexture(buffer, gl);

			VideoEffectUtil.ortho2D(gl, glu, newBounds.width, newBounds.height);

			gl.glMatrixMode(GL2.GL_MODELVIEW);
			gl.glTranslatef(left, top, 0);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, buffer.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, input.getTexture());
			gl.glEnable(GL2.GL_TEXTURE_RECTANGLE);

			gl.glColor4f(1, 1, 1, 1);

			int w = oldBounds.width;
			int h = oldBounds.height;

			gl.glBegin(GL2.GL_QUADS);
			gl.glTexCoord2f(0, 0);
			gl.glVertex2f(0, 0);
			gl.glTexCoord2f(w, 0);
			gl.glVertex2f(w, 0);
			gl.glTexCoord2f(w, h);
			gl.glVertex2f(w, h);
			gl.glTexCoord2f(0, h);
			gl.glVertex2f(0, h);
			gl.glEnd();

			gl.glDisable(GL2.GL_TEXTURE_RECTANGLE);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

			IVideoBuffer output = buffer;
			buffer = null;
			return output;

		} finally {
			input.dispose();
			if (buffer != null) buffer.dispose();
		}
	}

}
