/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


/**
 * 輝度算出用定数
 */
const vec3 lumaVec = vec3(0.299, 0.587, 0.114);

/**
 * ディザ合成用乱数
 */
const vec2 randVec = vec2(12.9898, 78.233);

float rand1(vec2 coord, float seed)
{
	return fract(sin(dot(coord, randVec) + seed) * 43758.5453);
}

/**
 * なし
 */
vec4 blend_none(vec4 pDst, vec4 pSrc, float opacity)
{
	return pSrc * opacity;
}

/**
 * 通常
 */
vec4 blend_normal(vec4 pDst, vec4 pSrc, float opacity)
{
	pSrc *= opacity;
	return pSrc + pDst*(1.0-pSrc.a);
}

/**
 * ディザ合成、ダイナミックディザ合成
 */
vec4 blend_dissolve(vec4 pDst, vec4 pSrc, float opacity, float seed)
{
	float rand = rand1(gl_TexCoord[0].st, seed);
	float a = pSrc.a*opacity;
	return (rand < a || a >= 1.0) ? vec4(pSrc.rgb/pSrc.a, 1.0) : pDst;
}

/**
 * 比較(暗)
 */
vec4 blend_darken(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	return vec4(min(uSrc3, pDst3 + min(uSrc3, cda))*sa + pDst3*csa, a);
}

/**
 * 乗算
 */
vec4 blend_multiply(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 pSrc3 = pSrc.rgb;

//	return vec4(uSrc3*(pDst3 + cda)*sa + pDst3*csa, a);			// 数式的にはこっちだけど、等価な下の式の方が計算量が少ない。
	return vec4(pSrc3*(pDst3 + cda)*opacity + pDst3*csa, a);
}

/**
 * 焼き込みカラー
 */
vec4 blend_color_burn(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	float r = uSrc3.r+uDst3.r<1.0 ? 0.0 : uSrc3.r>0.0 ? 1.0-(1.0-uDst3.r)/uSrc3.r : 1.0;
	float g = uSrc3.g+uDst3.g<1.0 ? 0.0 : uSrc3.g>0.0 ? 1.0-(1.0-uDst3.g)/uSrc3.g : 1.0;
	float b = uSrc3.b+uDst3.b<1.0 ? 0.0 : uSrc3.b>0.0 ? 1.0-(1.0-uDst3.b)/uSrc3.b : 1.0;

	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * 焼き込みリニア
 */
vec4 blend_linear_burn(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	return vec4((max(uSrc3+uDst3-1.0, 0.0)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * カラー比較(暗)
 */
vec4 blend_darker_color(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	float dy = dot(uDst3, lumaVec);
	float sy = dot(uSrc3, lumaVec);

	return vec4(((dy < sy ? uDst3 : uSrc3)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * 加算
 */
vec4 blend_add(vec4 pDst, vec4 pSrc, float opacity)
{
	float a = pSrc.a*opacity + pDst.a*(1.0 - pSrc.a*opacity);
	return vec4(pSrc.rgb*opacity + pDst.rgb, a);
}

/**
 * 比較(明)
 */
vec4 blend_lighten(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
//	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	return vec4(max(uSrc3, pDst3)*sa + pDst3*csa, a);
}

/**
 * スクリーン
 */
vec4 blend_screen(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	return vec4(((1.0-(1.0-uDst3)*(1.0-uSrc3))*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * 覆い焼きカラー
 */
vec4 blend_color_dodge(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	float r = uSrc3.r+uDst3.r>1.0 ? 1.0 : uDst3.r>0.0 ? uDst3.r/(1.0-uSrc3.r) : 0.0;
	float g = uSrc3.g+uDst3.g>1.0 ? 1.0 : uDst3.g>0.0 ? uDst3.g/(1.0-uSrc3.g) : 0.0;
	float b = uSrc3.b+uDst3.b>1.0 ? 1.0 : uDst3.b>0.0 ? uDst3.b/(1.0-uSrc3.b) : 0.0;

	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * 覆い焼きリニア
 */
vec4 blend_linear_dodge(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	return vec4((min(uSrc3+uDst3, 1.0)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * カラー比較(明)
 */
vec4 blend_lighter_color(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	float dy = dot(uDst3, lumaVec);
	float sy = dot(uSrc3, lumaVec);

	return vec4(((dy > sy ? uDst3 : uSrc3)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * オーバーレイ
 */
vec4 blend_overlay(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	float r = uDst3.r<0.5 ? 2.0*uSrc3.r*uDst3.r : 1.0-2.0*(1.0-uSrc3.r)*(1.0-uDst3.r);
	float g = uDst3.g<0.5 ? 2.0*uSrc3.g*uDst3.g : 1.0-2.0*(1.0-uSrc3.g)*(1.0-uDst3.g);
	float b = uDst3.b<0.5 ? 2.0*uSrc3.b*uDst3.b : 1.0-2.0*(1.0-uSrc3.b)*(1.0-uDst3.b);

	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * ソフトライト
 */
vec4 blend_soft_light(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);


	// TODO AEと完全には一致しない。

	float r = uSrc3.r<=0.5  ? uDst3.r-(1.0-2.0*uSrc3.r)*uDst3.r*(1.0-uDst3.r) :
			  uDst3.r<=0.25 ? uDst3.r+(2.0*uSrc3.r-1.0)*(((16.0*uDst3.r-12.0)*uDst3.r+4.0)*uDst3.r-uDst3.r) :
			                  uDst3.r+(2.0*uSrc3.r-1.0)*(sqrt(uDst3.r)-uDst3.r);
	float g = uSrc3.g<=0.5  ? uDst3.g-(1.0-2.0*uSrc3.g)*uDst3.g*(1.0-uDst3.g) :
			  uDst3.g<=0.25 ? uDst3.g+(2.0*uSrc3.g-1.0)*(((16.0*uDst3.g-12.0)*uDst3.g+4.0)*uDst3.g-uDst3.g) :
			                  uDst3.g+(2.0*uSrc3.g-1.0)*(sqrt(uDst3.g)-uDst3.g);
	float b = uSrc3.b<=0.5  ? uDst3.b-(1.0-2.0*uSrc3.b)*uDst3.b*(1.0-uDst3.b) :
			  uDst3.b<=0.25 ? uDst3.b+(2.0*uSrc3.b-1.0)*(((16.0*uDst3.b-12.0)*uDst3.b+4.0)*uDst3.b-uDst3.b) :
			                  uDst3.b+(2.0*uSrc3.b-1.0)*(sqrt(uDst3.b)-uDst3.b);

	// AEではこれに近い成分が加えられているようである。これは上の式でuDst3=0.5のもの。
	float r2 = uSrc3.r<=0.5 ? 0.5*uSrc3.r+0.25 : 0.414214*uSrc3.r+0.292893;
	float g2 = uSrc3.g<=0.5 ? 0.5*uSrc3.g+0.25 : 0.414214*uSrc3.g+0.292893;
	float b2 = uSrc3.b<=0.5 ? 0.5*uSrc3.b+0.25 : 0.414214*uSrc3.b+0.292893;

	return vec4((vec3(r, g, b)*da + vec3(r2, g2, b2)*cda)*sa + pDst3*csa, a);
}

/**
 * ハードライト
 */
vec4 blend_hard_light(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	float r = uSrc3.r<0.5 ? 2.0*uDst3.r*uSrc3.r : 1.0-2.0*(1.0-uDst3.r)*(1.0-uSrc3.r);
	float g = uSrc3.g<0.5 ? 2.0*uDst3.g*uSrc3.g : 1.0-2.0*(1.0-uDst3.g)*(1.0-uSrc3.g);
	float b = uSrc3.b<0.5 ? 2.0*uDst3.b*uSrc3.b : 1.0-2.0*(1.0-uDst3.b)*(1.0-uSrc3.b);

	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * リニアライト
 */
vec4 blend_linear_light(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	float r = uSrc3.r<0.5 ? (uDst3.r<1.0-uSrc3.r*2.0 ? 0.0 : uSrc3.r*2.0+uDst3.r-1.0)
						  : (uDst3.r<2.0-uSrc3.r*2.0 ? uSrc3.r*2.0+uDst3.r-1.0 : 1.0);
	float g = uSrc3.g<0.5 ? (uDst3.g<1.0-uSrc3.g*2.0 ? 0.0 : uSrc3.g*2.0+uDst3.g-1.0)
						  : (uDst3.g<2.0-uSrc3.g*2.0 ? uSrc3.g*2.0+uDst3.g-1.0 : 1.0);
	float b = uSrc3.b<0.5 ? (uDst3.b<1.0-uSrc3.b*2.0 ? 0.0 : uSrc3.b*2.0+uDst3.b-1.0)
						  : (uDst3.b<2.0-uSrc3.b*2.0 ? uSrc3.b*2.0+uDst3.b-1.0 : 1.0);

	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * ビビッドライト
 */
vec4 blend_vivid_light(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	float r = uSrc3.r<0.5 ? (uDst3.r<=1.0-uSrc3.r*2.0 ? 0.0 : (uDst3.r-(1.0-uSrc3.r*2.0))/(uSrc3.r*2.0))
						  : (uDst3.r< 2.0-uSrc3.r*2.0 ? uDst3.r/(2.0-uSrc3.r*2.0) : 1.0);
	float g = uSrc3.g<0.5 ? (uDst3.g<=1.0-uSrc3.g*2.0 ? 0.0 : (uDst3.g-(1.0-uSrc3.g*2.0))/(uSrc3.g*2.0))
						  : (uDst3.g< 2.0-uSrc3.g*2.0 ? uDst3.g/(2.0-uSrc3.g*2.0) : 1.0);
	float b = uSrc3.b<0.5 ? (uDst3.b<=1.0-uSrc3.b*2.0 ? 0.0 : (uDst3.b-(1.0-uSrc3.b*2.0))/(uSrc3.b*2.0))
						  : (uDst3.b< 2.0-uSrc3.b*2.0 ? uDst3.b/(2.0-uSrc3.b*2.0) : 1.0);

	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * ピンライト
 */
vec4 blend_pin_light(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	float r = uSrc3.r<0.5 ? min(uSrc3.r*2.0, uDst3.r) : max(uSrc3.r*2.0-1.0, uDst3.r);
	float g = uSrc3.g<0.5 ? min(uSrc3.g*2.0, uDst3.g) : max(uSrc3.g*2.0-1.0, uDst3.g);
	float b = uSrc3.b<0.5 ? min(uSrc3.b*2.0, uDst3.b) : max(uSrc3.b*2.0-1.0, uDst3.b);

	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * ハードミックス
 */
vec4 blend_hard_mix(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	float r = (uSrc3.r+uDst3.r>1.0 || uDst3.r >= 1.0) ? 1.0 : 0.0;
	float g = (uSrc3.g+uDst3.g>1.0 || uDst3.g >= 1.0) ? 1.0 : 0.0;
	float b = (uSrc3.b+uDst3.b>1.0 || uDst3.b >= 1.0) ? 1.0 : 0.0;

	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * 差
 */
vec4 blend_difference(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	return vec4((abs(uSrc3-uDst3)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * 除外
 */
vec4 blend_exclusion(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	float r = (1.0-uDst3.r)*uSrc3.r+(1.0-uSrc3.r)*uDst3.r;
	float g = (1.0-uDst3.g)*uSrc3.g+(1.0-uSrc3.g)*uDst3.g;
	float b = (1.0-uDst3.b)*uSrc3.b+(1.0-uSrc3.b)*uDst3.b;

	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * clip_color
 */
vec3 clip_color(vec3 color)
{
	float l = min(max(dot(color, lumaVec), 0.0), 1.0);
	float n = min(min(color.r, color.g), color.b);
	if (n < 0.0) {
		color = l + (color-l) * l / (l-n);
	}
	float x = max(max(color.r, color.g), color.b);
	if (x > 1.0) {
		color = l + (color-l) * (1.0-l) / (x-l);
	}
	return color;
}

/**
 * set_lum
 */
vec3 set_lum(vec3 color, float l)
{
	return clip_color(color + (l - dot(color, lumaVec)));
}

/**
 * sat
 */
float sat(vec3 color)
{
	return max(max(color.r, color.g), color.b) - min(min(color.r, color.g), color.b);
}

/**
 * set_sat
 */
vec3 set_sat(vec3 color, float s)
{
	float r = color.r;
	float g = color.g;
	float b = color.b;
	float n = min(min(r, g), b);
	float x = max(max(r, g), b);
	if (x > n) {
		if (x == r) {
			if (n == g) {
				b = (b-n) * s / (x-n);
				g = 0.0;
			} else {
				g = (g-n) * s / (x-n);
				b = 0.0;
			}
			r = s;
		} else if (x == g) {
			if (n == b) {
				r = (r-n) * s / (x-n);
				b = 0.0;
			} else {
				b = (b-n) * s / (x-n);
				r = 0.0;
			}
			g = s;
		} else {
			if (n == r) {
				g = (g-n) * s / (x-n);
				r = 0.0;
			} else {
				r = (r-n) * s / (x-n);
				g = 0.0;
			}
			b = s;
		}
		color = vec3(r, g, b);
	} else {
		color = vec3(0.0);
	}
	return color;
}

/**
 * 色相
 */
vec4 blend_hue(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	vec3 newDst3 = set_lum(set_sat(uSrc3, sat(uDst3)), dot(uDst3, lumaVec));

	return vec4((newDst3*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * 彩度
 */
vec4 blend_saturation(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	vec3 newDst3 = set_lum(set_sat(uDst3, sat(uSrc3)), dot(uDst3, lumaVec));

	return vec4((newDst3*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * カラー
 */
vec4 blend_color(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	vec3 newDst3 = set_lum(uSrc3, dot(uDst3, lumaVec));

	return vec4((newDst3*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * 輝度
 */
vec4 blend_luminosity(vec4 pDst, vec4 pSrc, float opacity)
{
	float da = pDst.a;
	float sa = pSrc.a*opacity;
	float cda = 1.0 - da;
	float csa = 1.0 - sa;
	float a = sa + da*csa;

	vec3 pDst3 = pDst.rgb;
	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);
	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);

	vec3 newDst3 = set_lum(uDst3, dot(uSrc3, lumaVec));

	return vec4((newDst3*da + uSrc3*cda)*sa + pDst3*csa, a);
}

/**
 * ステンシルアルファ
 */
vec4 blend_stencil_alpha(vec4 pDst, vec4 pSrc, float opacity)
{
	return pDst * (pSrc.a * opacity);
}

/**
 * ステンシルルミナンス
 */
vec4 blend_stencil_luma(vec4 pDst, vec4 pSrc, float opacity)
{
	return pDst * (dot(pSrc.rgb, lumaVec) * opacity);
}

/**
 * シルエットアルファ
 */
vec4 blend_silhouette_alpha(vec4 pDst, vec4 pSrc, float opacity)
{
	return pDst * (1.0 - pSrc.a * opacity);
}

/**
 * シルエットルミナンス
 */
vec4 blend_silhouette_luma(vec4 pDst, vec4 pSrc, float opacity)
{
	return pDst * (1.0 - dot(pSrc.rgb, lumaVec) * opacity);
}
