/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api;

/**
 * <code>AudioMode</code>列挙型は、オーディオの内部処理に使用可能なチャンネル数、
 * サンプルレート、サンプルサイズおよびデータ型の組み合わせをカプセル化したものです。
 */
public enum AudioMode {

	/**
	 * ステレオ / 48kHz / 16bit整数
	 */
	STEREO_48KHZ_INT16		(2, 48000, DataType.SHORT),

	/**
	 * ステレオ / 48kHz / 32bit整数
	 */
	STEREO_48KHZ_INT32		(2, 48000, DataType.INT),

	/**
	 * ステレオ / 48kHz / 32bit浮動小数
	 */
	STEREO_48KHZ_FLOAT32	(2, 48000, DataType.FLOAT),

	/**
	 * ステレオ / 96kHz / 16bit整数
	 */
	STEREO_96KHZ_INT16		(2, 96000, DataType.SHORT),

	/**
	 * ステレオ / 96kHz / 32bit整数
	 */
	STEREO_96KHZ_INT32		(2, 96000, DataType.INT),

	/**
	 * ステレオ / 96kHz / 32bit浮動小数
	 */
	STEREO_96KHZ_FLOAT32	(2, 96000, DataType.FLOAT);


	/**
	 * オーディオサンプルのデータ型を表す列挙型
	 */
	public enum DataType {

		/**
		 * short型
		 */
		SHORT,

		/**
		 * int型
		 */
		INT,

		/**
		 * float型
		 */
		FLOAT
	}


	/**
	 * チャンネル数
	 */
	public final int channels;

	/**
	 * サンプルレート
	 */
	public final int sampleRate;

	/**
	 * 1サンプルの継続時間
	 */
	public final Time sampleDuration;

	/**
	 * 1サンプルのバイト数
	 */
	public final int sampleSize;

	/**
	 * 1フレームのバイト数（＝ チャンネル数 × 1サンプルのバイト数）
	 */
	public final int frameSize;

	/**
	 * サンプルのデータ型
	 */
	public final DataType dataType;


	AudioMode(int channels, int sampleRate, DataType dataType) {
		this.channels = channels;
		this.sampleRate = sampleRate;
		this.sampleDuration = new Time(1L, sampleRate);
		this.sampleSize = (dataType == DataType.SHORT) ? 2 : 4;
		this.frameSize = channels * this.sampleSize;
		this.dataType = dataType;
	}

}
