/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api;

/**
 * <p><code>Color</code>クラスは色を表現します。
 * <code>Color</code>オブジェクトは、色のRGB成分およびアルファ値をdouble値で保持します。
 * 通常、値の範囲は0.0以上1.0以下ですが、範囲外の値を使用して<code>Color</code>オブジェクトを生成することも可能です。
 * メンバ変数 <code>rawRed</code>, <code>rawGreen</code>, <code>rawBlue</code>
 * および <code>rawAlpha</code> はコンストラクタの各引数の値で初期化されます。
 * コンストラクタの各引数に0.0より小さい値または1.0より大きい値を指定した場合、
 * メンバ変数 <code>r</code>, <code>g</code>, <code>b</code> および <code>a</code>
 * は、0.0または1.0で初期化されます。コンストラクタ引数のRGB成分が <code>NaN</code>
 * の場合、メンバ変数 <code>r</code>, <code>g</code>, <code>b</code> は0.0で初期化されます。
 * コンストラクタ引数のアルファ値が <code>NaN</code> の場合、メンバ変数 <code>a</code> は1.0で初期化されます。</p>
 * 
 * <p><code>Color</code>オブジェクトは不変です。
 * <code>Color</code>オブジェクトは作成したあとに変更できないため、共用することができます。</p>
 */
public final class Color {

	/**
	 * 無色透明を表します。
	 */
	public static final Color COLORLESS_TRANSPARENT = new Color(0, 0, 0, 0);

	/**
	 * 黒を表します。
	 */
	public static final Color BLACK = new Color(0, 0, 0, 1);

	/**
	 * 白を表します。
	 */
	public static final Color WHITE = new Color(1, 1, 1, 1);

	/**
	 * 50%のグレーを表します。
	 */
	public static final Color GRAY = new Color(0.5, 0.5, 0.5, 1);


	/**
	 * この色の赤成分です。
	 */
	public final double r;

	/**
	 * この色の緑成分です。
	 */
	public final double g;

	/**
	 * この色の青成分です。
	 */
	public final double b;

	/**
	 * この色のアルファ値です。
	 */
	public final double a;

	/**
	 * コンストラクタで指定された赤成分です。
	 */
	public final double rawRed;

	/**
	 * コンストラクタで指定された緑成分です。
	 */
	public final double rawGreen;

	/**
	 * コンストラクタで指定された青成分です。
	 */
	public final double rawBlue;

	/**
	 * コンストラクタで指定されたアルファ値です。
	 */
	public final double rawAlpha;


	/**
	 * RGB成分およびアルファ値を使って<code>Color</code>オブジェクトを生成します。
	 * 
	 * @param red   赤成分
	 * @param green 緑成分
	 * @param blue  青成分
	 * @param alpha アルファ値
	 */
	public Color(double red, double green, double blue, double alpha) {
		super();
		rawRed   = red;
		rawGreen = green;
		rawBlue  = blue;
		rawAlpha = alpha;

		r = Double.isNaN(red  ) ? 0 : Math.max(0, Math.min(1, red));
		g = Double.isNaN(green) ? 0 : Math.max(0, Math.min(1, green));
		b = Double.isNaN(blue ) ? 0 : Math.max(0, Math.min(1, blue));
		a = Double.isNaN(alpha) ? 1 : Math.max(0, Math.min(1, alpha)); 
	}

	/**
	 * RGB成分を使ってアルファ値1.0の<code>Color</code>オブジェクトを生成します。
	 * 
	 * @param red   赤成分
	 * @param green 緑成分
	 * @param blue  青成分
	 */
	public Color(double red, double green, double blue) {
		this(red, green, blue, 1);
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		long temp;
		temp = Double.doubleToLongBits(rawAlpha);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(rawBlue);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(rawGreen);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(rawRed);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		return result;
	}

	/**
	 * この <code>Color</code> オブジェクトとほかのオブジェクトを比較します。
	 * この比較は、メンバ変数 <code>rawRed</code>, <code>rawGreen</code>,
	 * <code>rawBlue</code> および <code>rawAlpha</code> の値によって行われることに留意してください。
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Color other = (Color) obj;
		if (Double.doubleToLongBits(rawAlpha) != Double
				.doubleToLongBits(other.rawAlpha))
			return false;
		if (Double.doubleToLongBits(rawBlue) != Double
				.doubleToLongBits(other.rawBlue))
			return false;
		if (Double.doubleToLongBits(rawGreen) != Double
				.doubleToLongBits(other.rawGreen))
			return false;
		if (Double.doubleToLongBits(rawRed) != Double
				.doubleToLongBits(other.rawRed))
			return false;
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return "Color [r=" + r + ", g=" + g + ", b=" + b + ", a=" + a + "]";
	}

}
