/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api.annotations;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import ch.kuramo.javie.api.ShaderType;

/**
 * <p>この注釈が付けられたフィールドは、プログラマブルシェーダのソースコードに関するものであると認識されます。</p>
 * 
 * <p>この注釈は、<code>String</code> 型もしくは <code>String[]</code> 型の
 * <code>public</code> なフィールドに付けることができます。
 * <code>String</code> 型の場合はフィールドの値がソースコードのリソースファイル名であるものとして処理されます。
 * <code>String[]</code> 型の場合はフィールドの値自体がソースコードであるものとして処理されます。</p>
 * 
 * <p>この注釈を付けられたフィールドが <code>static</code> でない場合、
 * そのフィールドを含むクラスをインスタンス化してからフィールドの値が読み出されます。
 * <i>TODO: この場合の使用方法の詳細を書く</i></p>
 * 
 * <p>シェーダには、この注釈が付けられたフィールドを含むクラスの完全修飾クラス名とフィールド名を . で連結した 
 * 「完全修飾クラス名.フィールド名」という形式の名前が付けられます。</p>
 * 
 * <p>{@link #program()} が true のシェーダはシェーダプログラムです。
 * そうでない場合はシェーダオブジェクトです。</p>
 * 
 * <p>シェーダプログラムは {@link ch.kuramo.javie.api.services.IShaderRegistry} サービスから
 * {@link ch.kuramo.javie.api.IShaderProgram} オブジェクトとして取り出す事ができます。
 * シェーダのコンパイル及びリンクは、<code>IShaderProgram</code>
 * を初めて使用するときに自動的に実行されます。</p>
 * 
 * <p>使用例：
 * <blockquote><pre>
 *  &#64;Effect
 *  public class InvertColor {
 *  
 *  	<b>// ソースコードをリソースファイルに記述する場合
 *  	//&#64;ShaderSource
 *  	//public static final String INVERT_COLOR = "invert_color.frag";</b>
 *  
 *  	<b>// ソースコードを直接記述する場合
 *  	&#64;ShaderSource
 *  	public static final String[] INVERT_COLOR</b> = {
 *  		"uniform sampler2D source;",
 *  		"",
 *  		"void main(void)",
 *  		"{",
 *  		"	vec4 prmult = texture2D(source, gl_TexCoord[0].st);",
 *  		"	if (prmult.a != 0.0) {",
 *  		"		vec3 unmult = prmult.rgb/prmult.a;",
 *  		"		unmult = 1.0 - unmult;",
 *  		"		gl_FragColor = vec4(unmult, 1.0) * prmult.a;",
 *  		"	} else {",
 *  		"		gl_FragColor = vec4(0.0);",
 *  		"	}",
 *  		"}"
 *  	};
 *  
 *  	private final IVideoEffectContext context;
 *  
 *  	private final IVideoRenderSupport support;
 *  
 *  	private final IShaderProgram program;
 *  
 *  	&#64;Inject
 *  	public InvertColor(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
 *  		this.context = context;
 *  		this.support = support;
 *  
 *  		<b>// シェーダのソースコードに対応する IShaderProgram オブジェクトを取得する
 *  		program = shaders.getProgram(InvertColor.class, "INVERT_COLOR");</b>
 *  	}
 *  
 *  	public IVideoBuffer doVideoEffect() {
 *  		IVideoBuffer source = context.doPreviousEffect();
 *  		if (source.getBounds().isEmpty()) {
 *  			return source;
 *  		}
 *  		try {
 *  			Set&lt;GLUniformData&gt; uniforms = new HashSet&lt;GLUniformData&gt;();
 *  			uniforms.add(new GLUniformData("source", 0));
 *  
 *  			<b>// シェーダプログラムを使用する</b>
 *  			return support.useShaderProgram(<b>program</b>, uniforms, null, source);
 *  
 *  		} finally {
 *  			source.dispose();
 *  		}
 *  	}
 *  }
 * </pre></blockquote>
 * </p>
 */
@Target(ElementType.FIELD)
@Retention(RetentionPolicy.RUNTIME)
public @interface ShaderSource {

	/**
	 * シェーダの種類を指定します。
	 * 
	 * @return シェーダの種類
	 */
	ShaderType type() default ShaderType.FRAGMENT_SHADER;

	/**
	 * シェーダがシェーダプログラムかどうかを指定します。
	 * 
	 * @return シェーダがシェーダプログラムの場合は true、そうでない場合は false
	 */
	boolean program() default true;

	/**
	 * シェーダがシェーダプログラムの場合、
	 * それに関連付けるシェーダオブジェクトを指定します。
	 * 
	 * @return シェーダプログラムに関連付けるシェーダオブジェクトの名前
	 */
	String[] attach() default {};

}
