/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api.services;

import java.util.List;

import ch.kuramo.javie.api.IVideoBuffer;

/**
 * 画像の累積（アキュムレーション）処理を支援するサービスです。
 * 
 * @since 0.5.0
 */
public interface IAccumulationSupport {

	/**
	 * {@link #accumulate(List, List, IVideoBuffer)} メソッドおよび
	 * {@link #accumulate(List, double, IVideoBuffer)} メソッドにおいて、
	 * 一度に処理できるソース画像の最大数を返します。この数を超えるソース画像を
	 * <code>accumulate</code> メソッドに渡した場合、複数回に分けて処理されます。
	 * ビデオメモリを効率よく使用するためには、この数を超えたソース画像を
	 * <code>accumulate</code> メソッドに渡すべきではありません。
	 * 
	 * @return <code>accumulate</code> メソッドで一度に処理できるソース画像の最大数
	 */
	int getMaxSourcesAtATime();

	/**
	 * @param srcBuffers 累積するソース画像のリスト。
	 * @param weights <code>srcBuffers</code> の各画像に対する、累積時の重み付け。
	 * @param dstBuffer 累積先となる画像。累積結果はこの画像に上書きされます。
	 */
	void accumulate(List<IVideoBuffer> srcBuffers, List<Double> weights, IVideoBuffer dstBuffer);

	/**
	 * @param srcBuffers 累積するソース画像のリスト。
	 * @param weight 累積時の重み付け。<code>srcBuffers</code> の全ての画像に同じ重み付けがなされます。
	 * @param dstBuffer 累積先となる画像。累積結果はこの画像に上書きされます。
	 */
	void accumulate(List<IVideoBuffer> srcBuffers, double weight, IVideoBuffer dstBuffer);

}
