/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api.services;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IVideoBuffer;

/**
 * アルファチャンネルの操作を支援するサービスです。
 * 
 * @since 0.5.0
 */
public interface IAlphaChannelSupport {

	/**
	 * 乗算済みアルファの画像を、非乗算済みアルファ（ストレートアルファ）に変換します。
	 * 
	 * @param buffer 変換対象となる乗算済みアルファの画像
	 * @return <code>buffer</code> を非乗算済みアルファ（ストレートアルファ）に変換した画像
	 */
	IVideoBuffer unmultiply(IVideoBuffer buffer);

	/**
	 * 非乗算済みアルファ（ストレートアルファ）の画像を、乗算済みアルファに変換します。
	 * 変換結果は元のバッファに上書きされます。
	 * 
	 * @param buffer 変換対象となる非乗算済みアルファの画像。また、変換結果はこのバッファに上書きされます。
	 */
	void premultiply(IVideoBuffer buffer);

	/**
	 * アルファチャンネルの内容を破棄し、1.0で埋めます。
	 * 結果は元のバッファに上書きされます。
	 * 
	 * @param buffer アルファチャンネルの内容を破棄する画像。また、結果はこのバッファに上書きされます。
	 */
	void discardAlpha(IVideoBuffer buffer);

	/**
	 * アルファチャンネルの内容をグレースケール画像に変換します。
	 * 変換結果は元のバッファに上書きされます。
	 * 
	 * @param buffer 変換対象となる画像。また、変換結果はこのバッファに上書きされます。
	 * @param color この色にアルファ値を乗算した値が変換結果のピクセル値となります。
	 */
	void alphaToGrayscale(IVideoBuffer buffer, Color color);

	/**
	 * 乗算済みアルファの画像にカラーマットを追加します。
	 * 結果は元のバッファに上書きされます。
	 * 
	 * @param buffer カラーマットの追加対象となる、乗算済みアルファの画像。また、結果はこのバッファに上書きされます。
	 * @param color 追加するカラーマットの色。
	 */
	void addColorMatte(IVideoBuffer buffer, Color color);

	/**
	 * カラーマットが加えられている乗算済みアルファの画像から、カラーマットを除去します。
	 * 結果は元のバッファに上書きされます。
	 * 
	 * @param buffer カラーマットの除去対象となる、乗算済みアルファの画像。また、結果はこのバッファに上書きされます。
	 * @param color 除去するカラーマットの色。
	 */
	void removeColorMatte(IVideoBuffer buffer, Color color);

}
