/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import java.io.File;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.WeakHashMap;

import net.arnx.jsonic.JSON;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.ui.IWorkbenchWindow;

import ch.kuramo.javie.app.actions.OpenProjectAction;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.core.Util;

public class RecentProjects {

	private static final String RECENT_PROJECTS = "RECENT_PROJECTS";

	private static final IPreferenceStore prefStore = Activator.getDefault().getPreferenceStore();

	private static final WeakHashMap<IWorkbenchWindow, WeakReference<MenuManager>> map = Util.newWeakHashMap();


	public static MenuManager createMenuManager(IWorkbenchWindow window) {
		MenuManager mm = new MenuManager("最近のプロジェクトを開く");
		map.put(window, new WeakReference<MenuManager>(mm));

		rebuildMenu(window, mm, loadEntries());
		return mm;
	}

	public static void add(ProjectManager pm) {
		if (pm.getFile() == null) {
			throw new IllegalArgumentException();
		}

		String id = pm.getProject().getId();
		String path;
		try {
			path = pm.getFile().getCanonicalPath();
		} catch (IOException e) {
			path = pm.getFile().getAbsolutePath();
		}

		List<List<String>> entries = loadEntries();
		for (Iterator<List<String>> it = entries.iterator(); it.hasNext(); ) {
			List<String> idAndPath = it.next();
			if (idAndPath.get(1).equals(path)) {
				it.remove();
			}
		}
		entries.add(0, Arrays.asList(id, path));

		if (entries.size() > 30) {
			entries = entries.subList(0, 30);
			Perspective.deleteOld(pm.getWorkbenchWindow().getWorkbench(), entries);
		}

		prefStore.setValue(RECENT_PROJECTS, JSON.encode(entries));

		IWorkbenchWindow window = pm.getWorkbenchWindow();
		WeakReference<MenuManager> mmRef = map.get(window);
		MenuManager mm = mmRef.get();
		if (mm != null) {
			rebuildMenu(window, mm, entries);
		}
	}

	private static List<List<String>> loadEntries() {
		String data = prefStore.getString(RECENT_PROJECTS);
		if (data.length() > 0) {
			List<?> list = JSON.decode(data, List.class);
			if (list.size() > 0 && list.get(0) instanceof List<?>
					&& ((List<?>) list.get(0)).get(0) instanceof String) {
				@SuppressWarnings("unchecked")
				List<List<String>> tmp = (List<List<String>>) list;
				return tmp;
			}
		}
		return Util.newList();
	}

	private static void rebuildMenu(final IWorkbenchWindow window, MenuManager mm, List<List<String>> entries) {
		mm.removeAll();

		if (entries.isEmpty()) {
			Action placeholder = new Action("最近のプロジェクトはありません") { };
			placeholder.setEnabled(false);
			mm.add(placeholder);
			return;
		}

		if (entries.size() > 10) {
			entries = entries.subList(0, 10);
		}

		for (List<String> idAndPath : entries) {
			String path = idAndPath.get(1);
			final File file =  new File(path);
			if (file.exists() && file.isFile()) {
				mm.add(new Action(path) {
					public void run() {
						if (file.exists() && file.isFile()) {
							OpenProjectAction.openProject(window, file);
						} else {
							// TODO ファイルが見つからないことを知らせるエラーダイアログ
						}
					}
				});
			}
		}
	}

}
