/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.actions;

import org.eclipse.core.commands.operations.IOperationHistoryListener;
import org.eclipse.core.commands.operations.IUndoableOperation;
import org.eclipse.core.commands.operations.OperationHistoryEvent;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.LabelRetargetAction;

import ch.kuramo.javie.app.CommandIds;
import ch.kuramo.javie.app.project.ProjectManager;

public class RedoAction extends LabelRetargetAction implements IOperationHistoryListener {

	private final IWorkbenchWindow window;

	private final IAction defaultHandler;


	public RedoAction(IWorkbenchWindow window) {
		super(CommandIds.REDO, "やり直し");
		this.window = window;

		window.getPartService().addPartListener(this);
		setActionDefinitionId(CommandIds.REDO);

		ISharedImages sharedImages = window.getWorkbench().getSharedImages();
		setImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_REDO));
		setDisabledImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_REDO_DISABLED));

		defaultHandler = new Action() {
			public void run() {
				ProjectManager pm = ProjectManager.forWorkbenchWindow(RedoAction.this.window);
				if (pm != null) {
					pm.redo();
				}
			}
		};

		updateDefaultHandlerState(null);

		PlatformUI.getWorkbench().getOperationSupport()
				.getOperationHistory().addOperationHistoryListener(this);
	}

	public void dispose() {
		PlatformUI.getWorkbench().getOperationSupport()
				.getOperationHistory().removeOperationHistoryListener(this);
		
		super.dispose();
	}

	protected void setActionHandler(IAction handler) {
		if (handler == null) {
			handler = defaultHandler;
		}
		super.setActionHandler(handler);
	}

	public void historyNotification(OperationHistoryEvent event) {
		IUndoableOperation op = event.getOperation();
		ProjectManager pm = ProjectManager.forWorkbenchWindow(window);

		if (!op.hasContext(pm.getUndoContext())) {
			return;
		}

		switch (event.getEventType()) {
			case OperationHistoryEvent.OPERATION_REMOVED:
			case OperationHistoryEvent.REDONE:
			case OperationHistoryEvent.UNDONE:
				updateDefaultHandlerState(pm);
				break;
		}
	}

	private void updateDefaultHandlerState(ProjectManager pm) {
		if (pm != null && pm.canRedo()) {
			defaultHandler.setEnabled(true);
			defaultHandler.setText(String.format("やり直し： %s", pm.getRedoLabel()));
		} else {
			defaultHandler.setEnabled(false);
			defaultHandler.setText("やり直し不可");
		}
	}

}
