/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.player;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.lang.reflect.InvocationTargetException;
import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import javax.media.opengl.GL2;
import javax.media.opengl.GLAutoDrawable;
import javax.media.opengl.GLEventListener;
import javax.media.opengl.awt.GLCanvas;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.ControlContribution;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.SWT;
import org.eclipse.swt.awt.SWT_AWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.services.GLGlobal;

import com.google.inject.Inject;

public class GLCanvasFactory {

	private static final GLCanvasFactory factory = new GLCanvasFactory();

	public static GLCanvasFactory getFactory() {
		return factory;
	}


	public static class GLCanvasRecord {
		public final ScrolledComposite scrolled;
		public final Composite embedded;
		public final Frame awtFrame;
		public final GLCanvas glCanvas;

		public GLCanvasRecord(
				ScrolledComposite scrolled, Composite embedded,
				Frame awtFrame, GLCanvas glCanvas) {
			super();
			this.scrolled = scrolled;
			this.embedded = embedded;
			this.awtFrame = awtFrame;
			this.glCanvas = glCanvas;
		}
	}


	private int _poolSize;

	private Queue<GLCanvasRecord> _pool;

	private Composite _holder;

	@Inject
	private GLGlobal _glGlobal;


	private GLCanvasFactory() {
		InjectorHolder.getInjector().injectMembers(this);
	}

	public void enterPoolMode(int poolSize) {
		if (_pool != null) {
			throw new IllegalStateException("already in pool mode");
		}
		if (poolSize <= 0) {
			throw new IllegalArgumentException("poolSize must be one or greater");
		}

		_poolSize = poolSize;
		_pool = new LinkedList<GLCanvasRecord>();
	}

	public boolean isPoolMode() {
		return (_pool != null);
	}

	public GLCanvasRecord getGLCanvas(Composite parent) {
		if (_pool != null) {
			GLCanvasRecord record = _pool.poll();
			if (record == null) {
				throw new JavieRuntimeException("no GLCanvas is available");
			}
			record.scrolled.setParent(parent);
			record.scrolled.setVisible(true);
			return record;

		} else {
			return newRecord(parent);
		}
	}

	public void releaseGLCanvas(GLCanvasRecord record) {
		if (_pool != null) {
			record.scrolled.setVisible(false);
			record.scrolled.setParent(_holder);
			_pool.offer(record);
		} else {
			record.scrolled.dispose();
		}
	}

	private GLCanvasRecord newRecord(Composite parent) {
		ScrolledComposite scrolled = new ScrolledComposite(parent, SWT.H_SCROLL | SWT.V_SCROLL);
		scrolled.setExpandHorizontal(true);
		scrolled.setExpandVertical(true);
		scrolled.setBackground(parent.getDisplay().getSystemColor(SWT.COLOR_DARK_GRAY));

		Composite embedded = new Composite(scrolled, SWT.NO_BACKGROUND | SWT.EMBEDDED);
		scrolled.setContent(embedded);

		Frame awtFrame = SWT_AWT.new_Frame(embedded);
		GridBagLayout gridBag = new GridBagLayout();
		awtFrame.setLayout(gridBag);
		awtFrame.setBackground(Color.GRAY);

		GLCanvas glCanvas = _glGlobal.createCanvas();
		gridBag.setConstraints(glCanvas, new GridBagConstraints());
		awtFrame.add(glCanvas);

		return new GLCanvasRecord(scrolled, embedded, awtFrame, glCanvas);
	}

	private void initPool(Composite parent) {
		if (_holder != null) {
			throw new IllegalStateException("pool is already initialized");
		}
		if (_pool == null) {
			throw new IllegalStateException("not in pool mode");
		}

		_holder = new Composite(parent, SWT.NULL);

		final StackLayout layout = new StackLayout();
		_holder.setLayout(layout);

		final Display display = _holder.getDisplay();
		final RGB bgRGB = display.getSystemColor(SWT.COLOR_DARK_GRAY).getRGB();

		final Queue<Control> controls = new ConcurrentLinkedQueue<Control>();

		for (int i = 0; i < _poolSize; ++i) {
			final GLCanvasRecord record = newRecord(_holder);
			record.glCanvas.setPreferredSize(new Dimension(1, 1));
			record.glCanvas.addGLEventListener(new GLEventListener() {
				public void init(GLAutoDrawable drawable) {}
				public void dispose(GLAutoDrawable drawable) {}
				public void reshape(GLAutoDrawable drawable, int x, int y, int width, int height) {}
				public void display(GLAutoDrawable drawable) {
					GL2 gl = drawable.getGL().getGL2();
					gl.glPushAttrib(GL2.GL_COLOR_BUFFER_BIT);
					try {
						gl.glClearColor(bgRGB.red/255f, bgRGB.green/255f, bgRGB.blue/255f, 1f);
						gl.glClear(GL2.GL_COLOR_BUFFER_BIT);
					} finally {
						gl.glPopAttrib();
					}

					record.glCanvas.removeGLEventListener(this);

					display.asyncExec(new Runnable() {
						public void run() {
							releaseGLCanvas(record);
							layout.topControl = controls.poll();
							_holder.layout();
						}
					});
				}
			});

			controls.offer(record.scrolled);
		}

		display.asyncExec(new Runnable() {
			public void run() {
				layout.topControl = controls.poll();
				_holder.layout();

				// _holder が画面上に見えていないと初期化が進まないようなので、一時的に最大化する。
				Shell shell = _holder.getShell();
				boolean maximized = shell.getMaximized();
				try {
					shell.setMaximized(true);

					ProgressMonitorDialog dialog = new ProgressMonitorDialog(shell);
					dialog.run(true, false, new IRunnableWithProgress() {
						public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
							int prev = controls.size();
							monitor.beginTask("GLCanvasを初期化中...", prev);
							while (!controls.isEmpty()) {
								int curr = controls.size();
								monitor.worked(prev-curr);
								prev = curr;
								Thread.sleep(10);
							}
							monitor.done();
						}
					});

				} catch (InvocationTargetException e) {
					throw new JavieRuntimeException(e);
				} catch (InterruptedException e) {
					throw new JavieRuntimeException(e);
				} finally {
					shell.setMaximized(maximized);
				}
			}
		});
	}

	public ControlContribution createControlContribution() {
		return new ControlContribution("GLCanvasFactory") {
			protected Control createControl(Composite parent) {
				initPool(parent);
				return _holder;
			}
		};
	}

}
