/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.TARangeSelector;
import ch.kuramo.javie.core.TASelector;
import ch.kuramo.javie.core.TAWigglySelector;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectElementFactory;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class AddTASelectorOperation extends ProjectOperation {

	private final String compId;

	private final String layerId;

	private final int animatorIndex;

	private final String data;


	@Inject
	private ProjectElementFactory elementFactory;

	@Inject
	private ProjectEncoder encoder;

	@Inject
	private ProjectDecoder decoder;


	public AddTASelectorOperation(ProjectManager pm,
			TextLayer layer, TextAnimator animator, Class<? extends TASelector> selectorClass) {

		super(pm, "セレクタの追加");
		InjectorHolder.getInjector().injectMembers(this);

		LayerComposition comp = pm.checkLayer(layer);
		compId = comp.getId();
		layerId = layer.getId();
		animatorIndex = layer.getTextAnimators().indexOf(animator);
		if (animatorIndex == -1) {
			throw new IllegalArgumentException("no such TextAnimator found");
		}

		TASelector selector;
		if (selectorClass == TARangeSelector.class) {
			setLabel("範囲セレクタの追加");
			selector = elementFactory.newTARangeSelector();
		} else if (selectorClass == TAWigglySelector.class) {
			setLabel("ウィグリーセレクタの追加");
			selector = elementFactory.newTAWigglySelector();
//		} else if (selectorClass == TAExpressionSelector.class) {
//			setLabel("エクスプレッションセレクタの追加");
//			selector = elementFactory.newTAExpressionSelector();
		} else {
			throw new RuntimeException("unknown TASelector class: " + selectorClass.getName());
		}
		selector.setName(uniqueSelectorNames(animator, selector.getName()));

		data = encoder.encodeElement(selector);
	}

	private String uniqueSelectorNames(TextAnimator animator, String prefix) {
		Set<String> names = Util.newSet();
		for (TASelector selector : animator.getSelectors()) {
			names.add(selector.getName());
		}

		int n = 1;
		String name;
		while (true) {
			name = prefix + " " + (n++);
			if (!names.contains(name)) {
				return name;
			}
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		TASelector selector;
		try {
			selector = decoder.decodeElement(data, TASelector.class);
		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding TASelector data", e);
		}

		LayerComposition comp = project.getComposition(compId);
		TextLayer layer = (TextLayer) comp.getLayer(layerId);
		TextAnimator animator = layer.getTextAnimators().get(animatorIndex);

		animator.getSelectors().add(selector);

		if (pm != null) {
			int selectorIndex = animator.getSelectors().size() - 1;
			pm.fireTASelectorsAdd(comp, new Object[][] { { layer, animatorIndex, selectorIndex } });
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(compId);
		TextLayer layer = (TextLayer) comp.getLayer(layerId);
		TextAnimator animator = layer.getTextAnimators().get(animatorIndex);

		int selectorIndex = animator.getSelectors().size() - 1;
		animator.getSelectors().remove(selectorIndex);

		if (pm != null) {
			pm.fireTASelectorsRemove(comp, new Object[][] { { layer, animatorIndex, selectorIndex } });
		}

		return Status.OK_STATUS;
	}

}
