/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.List;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;

public class ModifyLayerOutPointOperation extends ProjectOperation {

	private class Entry {
		final String layerId;
		final Time oldOutPoint;
		final Time newOutPoint;

		public Entry(String layerId, Time oldOutPoint, Time newOutPoint) {
			super();
			this.layerId = layerId;
			this.oldOutPoint = oldOutPoint;
			this.newOutPoint = newOutPoint;
		}
	}


	private final String _compId;

	private volatile List<Entry> _entries = Util.newList();


	public ModifyLayerOutPointOperation(
			ProjectManager projectManager, LayerComposition comp,
			Object[][] layersAndOutPoints, Time deltaTime, String relation) {

		super(projectManager, "アウトポイントの変更", relation);

		_noEffect = true;

		projectManager.checkComposition(comp);
		_compId = comp.getId();

		Time frameDuration = comp.getFrameDuration();

		for (Object[] o : layersAndOutPoints) {
			Layer layer = (Layer) o[0];
			Time newOutPoint = deltaTime.add((Time) o[1]);

			Time inPoint = layer.getInPoint();
			if (!newOutPoint.after(inPoint)) {
				newOutPoint = Time.fromFrameNumber(inPoint.toFrameNumber(frameDuration)+1, frameDuration);
			}

			if (layer instanceof MediaItemLayer && !LayerNature.isTimeRemapEnabled(layer)) {
				MediaInput input = ((MediaItemLayer) layer).getMediaInput();
				if (input != null) {
					Time duration = input.getDuration();
					if (duration != null) {
						Time startTime = layer.getStartTime();
						Time endTime = startTime.add(new Time((long)(duration.timeValue/Math.abs(layer.getRate())), duration.timeScale));
						if (newOutPoint.after(endTime)) {
							newOutPoint = endTime;
						}
					}
				}
			}

			Time oldOutPoint = layer.getOutPoint();
			_entries.add(new Entry(layer.getId(), oldOutPoint, newOutPoint));

			if (!newOutPoint.equals(oldOutPoint)) {
				_noEffect = false;
			}
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		modifyOutPoint(false, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		modifyOutPoint(true, project, pm);
		return Status.OK_STATUS;
	}

	private void modifyOutPoint(boolean undo, Project project, ProjectManager pm) {
		List<Layer> layers = null;
		if (pm != null) {
			layers = Util.newList();
		}

		LayerComposition comp = project.getComposition(_compId);

		for (Entry e : _entries) {
			Layer layer = comp.getLayer(e.layerId);
			layer.setOutPoint(undo ? e.oldOutPoint : e.newOutPoint);

			if (pm != null) {
				layers.add(layer);
			}
		}

		if (pm != null) {
			pm.fireLayerTimesChange(comp, layers);
		}
	}

	@Override
	protected boolean merge(ProjectOperation operation, IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) {

		if (pm == null) {
			// _entries はシャドウプロジェクトに反映される前に
			// 後続のマージ可能オペレーションで上書きされているかもしれない。
			// しかし、シャドウオペレーションのキューが全て実行された時点での値は一致するので問題ない。
			modifyOutPoint(false, project, null);
			return true;
		}

		if (!(operation instanceof ModifyLayerOutPointOperation)) {
			return false;
		}

		ModifyLayerOutPointOperation newOp = (ModifyLayerOutPointOperation) operation;
		if (!newOp._compId.equals(_compId) || newOp._entries.size() != _entries.size()) {
			return false;
		}

		List<Entry> mergedEntries = Util.newList();
		boolean noEffect = true;

		for (int i = 0, n = _entries.size(); i < n; ++i) {
			Entry oldEntry = _entries.get(i);
			Entry newEntry = newOp._entries.get(i);
			if (!newEntry.layerId.equals(oldEntry.layerId)) {
				return false;
			}
			mergedEntries.add(new Entry(oldEntry.layerId, oldEntry.oldOutPoint, newEntry.newOutPoint));
			if (!newEntry.newOutPoint.equals(oldEntry.oldOutPoint)) {
				noEffect = false;
			}
		}

		_entries = mergedEntries;
		modifyOutPoint(false, project, pm);

		_noEffect = noEffect;

		return true;
	}

}
