/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.ListIterator;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.TASelector;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.Util;

public class MoveTASelectorsOperation extends ProjectOperation {

	private final String _compId;

	private final List<int[]> _srcData;

	private final String _dstLayerId;

	private final int _dstAnimatorIndex;

	private final int _dstIndex;


	public MoveTASelectorsOperation(ProjectManager pm,
			List<Object[]> selectors, TextLayer dstLayer, TextAnimator dstAnimator, int moveBefore) {

		super(pm, "セレクタの移動");

		if (selectors.isEmpty()) {
			_noEffect = true;
			_compId = null;
			_srcData = null;
			_dstLayerId = null;
			_dstAnimatorIndex = 0;
			_dstIndex = 0;
			return;
		}

		LayerComposition comp = pm.checkLayer((TextLayer) selectors.get(0)[0]);
		if (pm.checkLayer(dstLayer) != comp) {
			throw new IllegalArgumentException();
		}

		_compId = comp.getId();
		_srcData = Util.newList();
		_dstLayerId = dstLayer.getId();
		_dstAnimatorIndex = dstLayer.getTextAnimators().indexOf(dstAnimator);

		int dstSelectorCount = dstAnimator.getSelectors().size();
		int dstIndex;
		if (moveBefore < 0 || moveBefore > dstSelectorCount) {
			dstIndex = moveBefore = dstSelectorCount;
		} else {
			dstIndex = moveBefore;
		}

		List<TASelector> tmp1 = Util.newList();

		for (Object[] o : selectors) {
			TextLayer srcLayer = (TextLayer) o[0];
			TextAnimator srcAnimator = (TextAnimator) o[1];
			TASelector selector = (TASelector) o[2];

			int layerIndex = comp.getLayers().indexOf(srcLayer);
			if (layerIndex == -1) {
				throw new IllegalArgumentException();
			}

			int animatorIndex = srcLayer.getTextAnimators().indexOf(srcAnimator);
			if (animatorIndex == -1) {
				throw new IllegalArgumentException();
			}

			int selectorIndex = srcAnimator.getSelectors().indexOf(selector);
			if (selectorIndex == -1) {
				throw new IllegalArgumentException();
			}

			_srcData.add(new int[] { layerIndex, animatorIndex, selectorIndex });
			if (srcAnimator == dstAnimator && selectorIndex < moveBefore) {
				--dstIndex;
			}

			tmp1.add(selector);
		}

		_dstIndex = dstIndex;

		Collections.sort(_srcData, new Comparator<int[]>() {
			public int compare(int[] o1, int[] o2) {
				int compare = o1[0] - o2[0];
				if (compare != 0) return compare;

				compare = o1[1] - o2[1];
				if (compare != 0) return compare;

				return o2[2] - o1[2]; // セレクタは逆順
			}
		});

		List<TASelector> tmp2 = Util.newList(dstAnimator.getSelectors());
		tmp2.removeAll(tmp1);
		tmp2.addAll(_dstIndex, tmp1);
		_noEffect = dstAnimator.getSelectors().equals(tmp2);
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);
		TextLayer dstLayer = (TextLayer) comp.getLayer(_dstLayerId);
		TextAnimator dstAnimator = dstLayer.getTextAnimators().get(_dstAnimatorIndex);

		List<Object[]> data1 = null, data2 = null;
		if (pm != null) {
			data1 = Util.newList();
			data2 = Util.newList();
		}

		List<TASelector> selectors = Util.newList();

		for (int[] srcData : _srcData) {
			TextLayer srcLayer = (TextLayer) comp.getLayers().get(srcData[0]);
			TextAnimator srcAnimator = srcLayer.getTextAnimators().get(srcData[1]);
			selectors.add(0, srcAnimator.getSelectors().remove(srcData[2]));
			if (pm != null) {
				data1.add(new Object[] { srcLayer, srcData[1], -1 } );
				data2.add(new Object[] { dstLayer, _dstAnimatorIndex, _dstIndex + data2.size() });
			}
		}

		if (pm != null) {
			pm.fireTASelectorsRemove(comp, data1.toArray(new Object[data1.size()][]));
		}

		dstAnimator.getSelectors().addAll(_dstIndex, selectors);
		if (pm != null) {
			pm.fireTASelectorsAdd(comp, data2.toArray(new Object[data2.size()][]));
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);
		TextLayer dstLayer = (TextLayer) comp.getLayer(_dstLayerId);
		TextAnimator dstAnimator = dstLayer.getTextAnimators().get(_dstAnimatorIndex);

		List<Object[]> data = null;
		if (pm != null) {
			data = Util.newList();
		}

		List<TASelector> sublist = dstAnimator.getSelectors().subList(_dstIndex, _dstIndex + _srcData.size());
		List<TASelector> selectors = Util.newList(sublist);
		sublist.clear();

		if (pm != null) {
			pm.fireTASelectorsRemove(comp, new Object[][] { new Object[] { dstLayer, _dstAnimatorIndex, -1 } });
		}

		for (ListIterator<int[]> it = _srcData.listIterator(_srcData.size()); it.hasPrevious(); ) {
			int[] srcData = it.previous();
			TextLayer srcLayer = (TextLayer) comp.getLayers().get(srcData[0]);
			TextAnimator srcAnimator = srcLayer.getTextAnimators().get(srcData[1]);
			srcAnimator.getSelectors().add(srcData[2], selectors.remove(0));
			if (pm != null) {
				data.add(new Object[] { srcLayer, srcData[1], srcData[2] });
			}
		}
		if (pm != null) {
			pm.fireTASelectorsAdd(comp, data.toArray(new Object[data.size()][]));
		}

		return Status.OK_STATUS;
	}

}
