/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class RemoveTextAnimatorsOperation extends ProjectOperation {

	private static class Entry {
		final String data;
		final String layerId;
		final int animatorIndex;

		private Entry(String data, String layerId, int animatorIndex) {
			this.data = data;
			this.layerId = layerId;
			this.animatorIndex = animatorIndex;
		}
	}


	private final String compId;

	private final List<Entry> entries = Util.newList();

	@Inject
	private ProjectEncoder encoder;

	@Inject
	private ProjectDecoder decoder;


	public RemoveTextAnimatorsOperation(ProjectManager pm, Collection<Object[]> animators) {
		super(pm, "テキストアニメータの削除");
		InjectorHolder.getInjector().injectMembers(this);

		LayerComposition comp = pm.checkLayer((TextLayer) animators.iterator().next()[0]);
		compId = comp.getId();

		for (Object[] o : animators) {
			TextLayer layer = (TextLayer) o[0];
			TextAnimator animator = (TextAnimator) o[1];

			if (pm.checkLayer(layer) != comp) {
				throw new IllegalArgumentException();
			}

			int animatorIndex = layer.getTextAnimators().indexOf(animator);
			if (animatorIndex == -1) {
				throw new IllegalArgumentException();
			}

			String data = encoder.encodeElement(animator);
			entries.add(new Entry(data, layer.getId(), animatorIndex));
		}

		Collections.sort(entries, new Comparator<Entry>() {
			public int compare(Entry o1, Entry o2) {
				int c = o1.layerId.compareTo(o2.layerId);	// レイヤー順序は重要ではない
				return (c != 0) ? c : o1.animatorIndex - o2.animatorIndex;
			}
		});
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(compId);

		// 後ろから処理しないとインデックスがずれるので。
		List<Entry> reverseEntries = Util.newList(entries);
		Collections.reverse(reverseEntries);

		for (Entry entry : reverseEntries) {
			TextLayer layer = (TextLayer) comp.getLayer(entry.layerId);
			layer.getTextAnimators().remove(entry.animatorIndex);
		}

		if (pm != null) {
			fireTextAnimatorsAddRemove(false, comp, pm);
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(compId);

		for (Entry entry : entries) {
			TextAnimator animator;
			try {
				animator = decoder.decodeElement(entry.data, TextAnimator.class);
				// TextAnimatorにafterDecodeメソッドは無い。
				//animator.afterDecode(project);
			} catch (ProjectDecodeException e) {
				throw new JavieRuntimeException(e);
			}
			TextLayer layer = (TextLayer) comp.getLayer(entry.layerId);
			layer.getTextAnimators().add(entry.animatorIndex, animator);
		}

		if (pm != null) {
			fireTextAnimatorsAddRemove(true, comp, pm);
		}

		return Status.OK_STATUS;
	}

	private Object[][] createData(LayerComposition comp) {
		List<Object[]> data = Util.newList();

		for (Entry entry : entries) {
			TextLayer layer = (TextLayer) comp.getLayer(entry.layerId);
			data.add(new Object[] { layer, entry.animatorIndex });	// TODO 削除のときはインデックスを -1 などにしておいた方がいいかも。
		}

		return data.toArray(new Object[data.size()][]);
	}

	private void fireTextAnimatorsAddRemove(boolean add, LayerComposition comp, ProjectManager pm) {
		Object[][] data = createData(comp);

		if (add) {
			pm.fireTextAnimatorsAdd(comp, data);
		} else {
			pm.fireTextAnimatorsRemove(comp, data);
		}
	}

}
