/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.util.List;
import java.util.Map;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.TreeItem;

import ch.kuramo.javie.app.EnumLabels;
import ch.kuramo.javie.app.project.NewTextAnimatorOperation;
import ch.kuramo.javie.app.project.PerCharacter3DOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.TAProperty;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.Util;

public class TextAnimatorsElement extends Element {

	private static final int[] UNDERLINE_DASH = new int[] { 2, 2 };

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");

	private static final boolean WIN32 = SWT.getPlatform().equals("win32");


	private Rectangle valueArea;

	private final List<Element> children = Util.newList();

	private TAOptionsElement optionsElement;

	private final Map<TextAnimator, TextAnimatorElement> animatorElementMap = Util.newMap();


	public TextAnimatorsElement(TextElement parent) {
		super(parent);
	}

	public String getColumnText(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return "テキストアニメータ";

			case LayerCompositionView.VALUE_COL:
				// ここでnullや空文字を返すと、なぜかCellEditorがすぐにフォーカスアウトしてしまう場合がある。
				return " ";

			default:
				return null;
		}
	}

	@Override
	public void paintColumn(Event event) {
		if (event.index == LayerCompositionView.VALUE_COL) {
			String valueStr = ">";

			GC gc = event.gc;
			int x = event.x + 5;
			int y = event.y;
			int height = ((TreeItem) event.item).getBounds(event.index).height;
			boolean focused = ((event.detail & SWT.SELECTED) != 0 && ((Control) event.widget).isFocusControl());


			gc.setForeground(gc.getDevice().getSystemColor(
					focused ? SWT.COLOR_WHITE : SWT.COLOR_DARK_BLUE));

			Point extent = gc.textExtent(valueStr, SWT.DRAW_TRANSPARENT);

			y += (height - extent.y) / 2;
			gc.drawString(valueStr, x, y, true);

			valueArea = new Rectangle(x, y, extent.x, extent.y);

			y += extent.y;
			gc.setLineDash(UNDERLINE_DASH);
			if (COCOA) {
				gc.drawLine(x, y - 1, x + extent.x, y - 1);
			} else if (WIN32) {
				gc.drawLine(x, y, x + extent.x - 2, y);
			} else {
				gc.drawLine(x, y, x + extent.x, y);
			}

		} else {
			super.paintColumn(event);
		}
	}

	@Override
	public void updateCursor(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.VALUE_COL:
				Cursor cursor = null;
				if (valueArea != null && valueArea.contains(event.x, event.y)) {
					cursor = event.display.getSystemCursor(SWT.CURSOR_HAND);
				}
				viewer.getTree().setCursor(cursor);
				break;

			default:
				super.updateCursor(event, columnIndex);
				break;
		}
	}

	@Override
	public void mouseDown(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.VALUE_COL:
				if (event.button == 1 && valueArea != null && valueArea.contains(event.x, event.y)) {
					showMenu();
				}
				break;

			default:
				super.mouseDown(event, columnIndex);
				break;
		}
	}

	private void showMenu() {
		Menu menu = createPopUpMenu(valueArea.x-5, valueArea.y/*+valueArea.height+2*/);

		final TextLayer layer = getLayer();
		boolean perCharacter3D = LayerNature.isThreeD(layer) && layer.isPerCharacter3D();

		Action action = new Action("文字単位の3D化", IAction.AS_CHECK_BOX) {
			public void run() {
				ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
				pm.postOperation(PerCharacter3DOperation.toggle(pm, layer));
			}
		};
		action.setChecked(perCharacter3D);
		ActionContributionItem item = new ActionContributionItem(action);
		item.fill(menu, -1);

		new MenuItem(menu, SWT.SEPARATOR);

		Map<TAProperty, Action> propActions = Util.newMap();
		for (TAProperty p : TAProperty.values()) {
			final TAProperty[] properties;
			switch (p) {
				case rotationX:
				case rotationY:
				case skewAxis:
				case characterRange:
					continue;
				case rotationZ:
					if (perCharacter3D) {
						properties = new TAProperty[] { TAProperty.rotationX, TAProperty.rotationY, TAProperty.rotationZ };
					} else {
						properties = new TAProperty[] { TAProperty.rotationZ };
					}
					break;
				case skew:
					properties = new TAProperty[] { TAProperty.skew, TAProperty.skewAxis };
					break;
				case characterValue:
				case characterOffset:
					properties = new TAProperty[] { TAProperty.characterRange, p };
					break;
				default:
					properties = new TAProperty[] { p };
					break;
			}
			action = new Action(EnumLabels.getLabel(p)) {
				public void run() {
					ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
					pm.postOperation(new NewTextAnimatorOperation(pm, layer, properties));
				}
			};
			propActions.put(p, action);
		}
		TextAnimatorElement.createPropertyMenuItems(menu, propActions);

		menu.setVisible(true);
	}

	@Override
	public boolean hasChildren() {
		return !getLayer().getTextAnimators().isEmpty();
	}

	@Override
	public Element[] getChildren() {
		prepareChildren();
		return children.toArray(new Element[children.size()]);
	}

	TextLayer getLayer() {
		return ((TextElement) parent).getLayer();
	}

	private TextAnimatorElement getTextAnimatorElement(TextAnimator animator) {
		TextAnimatorElement element = animatorElementMap.get(animator);
		if (element == null) {
			element = new TextAnimatorElement(this, animator);
			animatorElementMap.put(animator, element);
		}
		return element;
	}

	private void prepareChildren() {
		children.clear();

		TextLayer layer = getLayer();

		if (optionsElement == null) {
			optionsElement = new TAOptionsElement(this);
		}
		children.add(optionsElement);

		for (TextAnimator animator : layer.getTextAnimators()) {
			children.add(getTextAnimatorElement(animator));
		}

		animatorElementMap.values().retainAll(children);
	}

}
