/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.util.regex.Pattern;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.app.widgets.InPlaceEditorShell;
import ch.kuramo.javie.core.AnimatableString;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.Util;

public class TextSourceElement extends LayerAnimatableValueElement.ForSimpleProperty {

	public TextSourceElement(Element parent, TextLayer layer) {
		super(parent, layer, "sourceText", "ソーステキスト");
	}

	protected AnimatableValueElementDelegate<?> createDelegate() {
		AnimatableString avalue = PropertyUtil.getProperty(layer, getProperty());
		return new TextSourceElementDelegate(this, name, avalue);
	}

}

class TextSourceElementDelegate extends AnimatableStringElementDelegate {

	private static final Pattern LF = Pattern.compile("\\n");
	private static final Pattern CR_OR_CRLF = Pattern.compile("\\r\\n?");

	private Point editorSize = new Point(300, 200);


	public TextSourceElementDelegate(
			AnimatableValueElement element, String name, AnimatableString avalue) {

		super(element, name, avalue);
	}

	protected void openInPlaceEditor() {
		final Shell shell = InPlaceEditorShell.create(
				element.viewer.getTree(), valueArea.get(valueIndex),
				editorSize.x, editorSize.y, true);

		shell.setMinimumSize(100, 100);

		FillLayout fillLayout = new FillLayout();
		fillLayout.marginWidth = 5;
		fillLayout.marginHeight = 5;
		shell.setLayout(fillLayout);

		final Text text = new Text(shell, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		text.setTabs(4);

		String str = LF.matcher(valueWithoutExpr).replaceAll(Text.DELIMITER);
		text.setText(str);
		text.selectAll();

		text.addModifyListener(new ModifyListener() {
			private final String relation = Util.randomId();
			public void modifyText(ModifyEvent e) {
				String value = CR_OR_CRLF.matcher(text.getText()).replaceAll("\n");
				modifyValue(value, relation);
			}
		});

		shell.addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				editorSize = shell.getSize();
			}
		});

		shell.open();
	}

	public boolean canEdit(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			return false;
		}
		return super.canEdit(columnIndex);
	}

}
