/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.mmd2javie;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.ListIterator;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.ContextFactory;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.ScriptableObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;

class MMD2Javie {

	private static final Logger logger = LoggerFactory.getLogger(MMD2Javie.class);


	private final ContextFactory contextFactory;

	MMD2Javie(ContextFactory contextFactory) {
		this.contextFactory = contextFactory;
	}

	boolean run(final File file, final Project project, IRunnableContext rc) {
		IRunnableWithProgress runnable = new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
				try {
					evaluateJSXFile(file, project, monitor);
				} catch (IOException e) {
					throw new InvocationTargetException(e);
				}
			}
		};

		try {
			logger.info("処理を開始しました...");
			rc.run(true, true, runnable);
			logger.info("処理を完了しました");
			return true;
		} catch (InterruptedException ex) {
			logger.info("処理を中断しました");
		} catch (InvocationTargetException ex) {
			Throwable t = ex.getTargetException();
			logger.error("エラーが発生しました", (t != null) ? t : ex);
		}
		return false;
	}

	private void evaluateJSXFile(File file, Project project, IProgressMonitor monitor) throws IOException, InterruptedException {
		List<String> lines = Util.newList();

		monitor.beginTask("JSXファイルを読み込み中...", IProgressMonitor.UNKNOWN);
		BufferedReader jsxReader = null;
		try {
			jsxReader = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF-8"));
			for (String line; (line = jsxReader.readLine()) != null; ) {
				lines.add(line);
			}
		} finally {
			monitor.done();
			if (jsxReader != null) jsxReader.close();
		}

		monitor.beginTask("スクリプトを処理中...", lines.size());
		try {
			Context context = contextFactory.enterContext();

			Scriptable scope = context.initStandardObjects();
			Object app = Context.javaToJS(AEScriptObjects.createAppObject(project), scope);
			ScriptableObject.putProperty(scope, "app", app);

			String sourceName = file.getName();
			StringBuilder sb = new StringBuilder();
			for (ListIterator<String> it = lines.listIterator(); it.hasNext(); ) {
				if (monitor.isCanceled()) {
					throw new InterruptedException("canceled");
				}

				int i = it.nextIndex();
				int j;

				sb.setLength(0);

				// スクリプトが長すぎると「64kリミットを超えた」というエラーが発生して実行できないので、
				// 適当な行数毎に実行する。2000行程度まで大丈夫なようだが、念のため200行にしている。
				for (j = 0; j < 200 && it.hasNext(); ++j) {
					sb.append(it.next()).append("\n");
				}

				context.evaluateString(scope, sb.toString(), sourceName, i, null);

				monitor.worked(j);
			}
		} finally {
			monitor.done();
			Context.exit();
		}
	}

}
