/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.FloatBuffer;
import java.nio.ShortBuffer;
import java.util.Arrays;
import java.util.concurrent.locks.ReentrantLock;

import javax.media.opengl.GL2;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderContext;

public class VideoBufferImpl implements IVideoBuffer {

	private static final Logger logger = LoggerFactory.getLogger(VideoBufferImpl.class);


	private final ColorMode colorMode;

	private final VideoBounds bounds;

	private int texture;

	private IArray<?> array;

	private final RenderContext context;

	private final IVideoRenderSupport support;

	private final IArrayPools arrayPools;

	private final GLGlobal glGlobal;


	public VideoBufferImpl(
			ColorMode colorMode, VideoBounds bounds,
			RenderContext context, IVideoRenderSupport support,
			IArrayPools arrayPools, GLGlobal glGlobal) {

		this.colorMode = colorMode;
		this.bounds = bounds;
		this.context = context;
		this.support = support;
		this.arrayPools = arrayPools;
		this.glGlobal = glGlobal;
	}

	@Override
	protected void finalize() throws Throwable {
		if (texture != 0 || array != null) {
			// TODO 解放忘れの場合どうする？
			//		finalize は別のスレッドから呼ばれるので、ここでテクスチャの解放をすることはできない。
			//		生成時に RenderContext に登録しておいて、
			//		RenderContext の deactivate で検出／解放するといいかも。
			logger.warn("finalizing a VideoBufferImpl object, but the object is not disposed.");

			if (array != null) {
				array.release();
				array = null;
			}
		}

		super.finalize();
	}

	public void dispose() {
		if (texture != 0) {
			deleteTexture(texture);
			texture = 0;
		}

		if (array != null) {
			array.release();
			array = null;
		}
	}

	public boolean isTexture() {
		return (texture != 0);
	}

	public boolean isArray() {
		return (array != null);
	}

	public int getTexture() {
		if (texture == 0) {
			texture = createTexture(colorMode, bounds.width, bounds.height);

			// 配列にデータがある場合、テクスチャにコピーして配列を解放する。
			if (array != null) {
				copyArrayToTexture(array.getArray(), texture, colorMode, bounds.width, bounds.height);
				array.release();
				array = null;
			}

		} else if (array != null) {
			// bug?
			throw new IllegalStateException("both texture and array exist.");
		}

		return texture;
	}

	public Object getArray() {
		if (array == null) {
			array = createArray(colorMode, bounds.width, bounds.height);

			// テクスチャにデータがある場合、配列にコピーしてテクスチャを破棄する
			if (texture != 0) {
				copyTextureToArray(texture, array.getArray(), colorMode, bounds.width, bounds.height);
				deleteTexture(texture);
				texture = 0;
			}

		} else if (texture != 0) {
			// bug?
			throw new IllegalStateException("both texture and array exist.");
		}

		return array.getArray();
	}

	public ColorMode getColorMode() {
		return colorMode;
	}

	public VideoBounds getBounds() {
		return bounds;
	}

	public TextureFilter getTextureFilter() {
		int texture = getTexture();
		GL2 gl = context.getGL().getGL2();

		int[] params = new int[2];
		int[] current = new int[1];
		gl.glGetIntegerv(GL2.GL_TEXTURE_BINDING_2D, current, 0);
		try {
			gl.glBindTexture(GL2.GL_TEXTURE_2D, texture);
			gl.glGetTexParameteriv(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MIN_FILTER, params, 0);
			gl.glGetTexParameteriv(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MAG_FILTER, params, 1);
		} finally {
			gl.glBindTexture(GL2.GL_TEXTURE_2D, current[0]);
		}

		switch (params[0]) {
			case GL2.GL_NEAREST:
				if (params[1] != GL2.GL_NEAREST) throw new IllegalStateException();
				return TextureFilter.NEAREST;

			case GL2.GL_LINEAR:
				if (params[1] != GL2.GL_LINEAR) throw new IllegalStateException();
				return TextureFilter.LINEAR;

			case GL2.GL_LINEAR_MIPMAP_LINEAR:
				if (params[1] != GL2.GL_LINEAR) throw new IllegalStateException();
				return TextureFilter.MIPMAP;

			default:
				throw new IllegalStateException("unsupported filter: " + params[0]);
		}
	}

	public void setTextureFilter(TextureFilter filter) {
		int texture = getTexture();
		GL2 gl = context.getGL().getGL2();

		int[] current = new int[1];
		gl.glGetIntegerv(GL2.GL_TEXTURE_BINDING_2D, current, 0);
		try {
			gl.glBindTexture(GL2.GL_TEXTURE_2D, texture);

			switch (filter) {
				case NEAREST:
					gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_NEAREST);
					gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_NEAREST);
					break;

				case LINEAR:
					gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_LINEAR);
					gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_LINEAR);
					break;

				case MIPMAP:
					gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_LINEAR);

					int[] min = new int[1];
					gl.glGetTexParameteriv(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MIN_FILTER, min, 0);
					if (min[0] != GL2.GL_LINEAR_MIPMAP_LINEAR) {
						gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_LINEAR_MIPMAP_LINEAR);
						gl.glTexParameterf(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MAX_ANISOTROPY_EXT, 16.0f);
						gl.glGenerateMipmap(GL2.GL_TEXTURE_2D);
					}
					break;

				default:
					throw new UnsupportedOperationException("unsupported filter: " + filter.name());
			}

		} finally {
			gl.glBindTexture(GL2.GL_TEXTURE_2D, current[0]);
		}
	}

	public TextureWrapMode getTextureWrapMode() {
		int texture = getTexture();
		GL2 gl = context.getGL().getGL2();

		int[] params = new int[2];
		int[] current = new int[1];
		gl.glGetIntegerv(GL2.GL_TEXTURE_BINDING_2D, current, 0);
		try {
			gl.glBindTexture(GL2.GL_TEXTURE_2D, texture);
			gl.glGetTexParameteriv(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_WRAP_S, params, 0);
			gl.glGetTexParameteriv(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_WRAP_T, params, 1);
		} finally {
			gl.glBindTexture(GL2.GL_TEXTURE_2D, current[0]);
		}

		if (params[0] != params[1]) {
			throw new IllegalStateException("wrap mode of GL_TEXTURE_WRAP_S and GL_TEXTURE_WRAP_T differ");
		}
		switch (params[0]) {
			case GL2.GL_CLAMP_TO_BORDER:
				return TextureWrapMode.CLAMP_TO_BORDER;
			case GL2.GL_CLAMP_TO_EDGE:
				return TextureWrapMode.CLAMP_TO_EDGE;
			case GL2.GL_REPEAT:
				return TextureWrapMode.REPEAT;
			case GL2.GL_MIRRORED_REPEAT:
				return TextureWrapMode.MIRRORED_REPEAT;
			default:
				throw new IllegalStateException("unsupported wrap mode: " + params[0]);
		}
	}

	public void setTextureWrapMode(TextureWrapMode wrapMode) {
		int param;
		switch (wrapMode) {
			case CLAMP_TO_BORDER:
				param = GL2.GL_CLAMP_TO_BORDER;
				break;
			case CLAMP_TO_EDGE:
				param = GL2.GL_CLAMP_TO_EDGE;
				break;
			case REPEAT:
				param = GL2.GL_REPEAT;
				break;
			case MIRRORED_REPEAT:
				param = GL2.GL_MIRRORED_REPEAT;
				break;
			default:
				throw new UnsupportedOperationException("unsupported wrap mode: " + wrapMode.name());
		}

		int texture = getTexture();
		GL2 gl = context.getGL().getGL2();

		int[] current = new int[1];
		gl.glGetIntegerv(GL2.GL_TEXTURE_BINDING_2D, current, 0);
		try {
			gl.glBindTexture(GL2.GL_TEXTURE_2D, texture);
			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_WRAP_S, param);
			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_WRAP_T, param);
		} finally {
			gl.glBindTexture(GL2.GL_TEXTURE_2D, current[0]);
		}
	}

	public void clear() {
		clear(Color.COLORLESS_TRANSPARENT);
	}

	public void clear(Color color) {
		if (array != null) {
			clearArray(array, color);
		} else {
			clearTexture(color);
		}
	}

	public void copyToTexture(Object srcArray, ColorMode srcColorMode) {
		if (array != null) {
			array.release();
			array = null;
		}
		copyArrayToTexture(srcArray, getTexture(), srcColorMode, bounds.width, bounds.height);
	}


	private static final float[] FLOAT0000 = new float[] { 0, 0, 0, 0 };

	private int createTexture(ColorMode colorMode, int width, int height) {
		GL2 gl = context.getGL().getGL2();

		int[] texture = new int[1];
		int[] current = new int[1];
		gl.glGetIntegerv(GL2.GL_TEXTURE_BINDING_2D, current, 0);
		try {
			gl.glGenTextures(1, texture, 0);
			gl.glBindTexture(GL2.GL_TEXTURE_2D, texture[0]);

			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_NEAREST);
			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_NEAREST);
			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_BORDER);
			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_BORDER);
			gl.glTexParameterfv(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_BORDER_COLOR, FLOAT0000, 0);
			gl.glTexImage2D(GL2.GL_TEXTURE_2D, 0, colorMode.glInternalFormat,
					Math.max(1, width), Math.max(1, height), 0, GL2.GL_BGRA, colorMode.glDataType, null);

			int result = texture[0];
			texture[0] = 0;
			return result;

		} finally {
			gl.glBindTexture(GL2.GL_TEXTURE_2D, current[0]);
			if (texture[0] != 0) gl.glDeleteTextures(1, texture, 0);
		}
	}

	private void deleteTexture(int texture) {
		GL2 gl = context.getGL().getGL2();
		gl.glDeleteTextures(1, new int[] { texture }, 0);
	}

	private IArray<?> createArray(ColorMode colorMode, int width, int height) {
		int arraySize = width * height * 4;

		switch (colorMode) {
			case RGBA8:
				return arrayPools.getByteArray(arraySize);
			case RGBA16:
				return arrayPools.getShortArray(arraySize);
			case RGBA16_FLOAT:
			case RGBA32_FLOAT:
				return arrayPools.getFloatArray(arraySize);
			default:
				throw new UnsupportedOperationException("ColorMode: " + colorMode.name());
		}
	}

	private void copyArrayToTexture(Object srcArray, int dstTexture, ColorMode srcColorMode, int width, int height) {
		ReentrantLock lock = glGlobal.getGlobalLock();
		lock.lock();
		try {

			GL2 gl = context.getGL().getGL2();

			int[] pbo = new int[1];
			int[] current = new int[2];
			gl.glGetIntegerv(GL2.GL_PIXEL_UNPACK_BUFFER_BINDING, current, 0);
			gl.glGetIntegerv(GL2.GL_TEXTURE_BINDING_2D, current, 1);
			try {
				gl.glGenBuffers(1, pbo, 0);
				gl.glBindBuffer(GL2.GL_PIXEL_UNPACK_BUFFER, pbo[0]);
				gl.glBindTexture(GL2.GL_TEXTURE_2D, dstTexture);

				int imageBytes = width * height * srcColorMode.javaPixelBytes;

				// 配列 -> PBO
				gl.glBufferData(GL2.GL_PIXEL_UNPACK_BUFFER, imageBytes, toNioBuffer(srcArray), GL2.GL_STREAM_DRAW);

				// PBO -> テクスチャ
				gl.glTexSubImage2D(GL2.GL_TEXTURE_2D, 0, 0, 0, width, height,
						GL2.GL_BGRA, srcColorMode.glDataType, 0);

			} finally {
				gl.glBindTexture(GL2.GL_TEXTURE_2D, current[1]);
				gl.glBindBuffer(GL2.GL_PIXEL_UNPACK_BUFFER, current[0]);
				if (pbo[0] != 0) gl.glDeleteBuffers(1, pbo, 0);
			}

		} finally {
			lock.unlock();
		}
	}

	private Buffer toNioBuffer(Object array) {
		if (array instanceof byte[]) {
			return ByteBuffer.wrap((byte[]) array);
			
		} else if (array instanceof short[]) {
			return ShortBuffer.wrap((short[]) array);

		} else if (array instanceof float[]) {
			return FloatBuffer.wrap((float[]) array);

		} else {
			throw new IllegalArgumentException(
					"not array or unsupported array type: "
					+ array.getClass().getName());
		}
	}

	private void copyTextureToArray(int srcTexture, Object dstArray, ColorMode srcColorMode, int width, int height) {
		ReentrantLock lock = glGlobal.getGlobalLock();
		lock.lock();
		try {

			GL2 gl = context.getGL().getGL2();

			int[] fb = new int[1];
			gl.glGetFramebufferAttachmentParameteriv(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME, fb, 0);
			if (fb[0] != 0) {
				throw new IllegalStateException("framebuffer is in use");
			}

			int[] pbo = new int[1];
			int[] current = new int[2];
			gl.glGetIntegerv(GL2.GL_PIXEL_PACK_BUFFER_BINDING, current, 0);
			gl.glGetIntegerv(GL2.GL_READ_BUFFER, current, 1);
			try {
				gl.glGenBuffers(1, pbo, 0);
				gl.glBindBuffer(GL2.GL_PIXEL_PACK_BUFFER, pbo[0]);

				int imageBytes = width * height * srcColorMode.javaPixelBytes;

				// PBOの領域を確保
				gl.glBufferData(GL2.GL_PIXEL_PACK_BUFFER, imageBytes, null, GL2.GL_STREAM_READ);

				// テクスチャをフレームバッファに割当てる
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_2D, srcTexture, 0);

				// フレームバッファ(テクスチャ) -> PBO
				gl.glReadBuffer(GL2.GL_COLOR_ATTACHMENT0);
				gl.glReadPixels(0, 0, width, height, GL2.GL_BGRA, srcColorMode.glDataType, 0);

				// PBO -> 配列
				ByteBuffer mappedPBO = gl.glMapBuffer(GL2.GL_PIXEL_PACK_BUFFER, GL2.GL_READ_ONLY);
				copyPBOToArray(mappedPBO, dstArray);
				gl.glUnmapBuffer(GL2.GL_PIXEL_PACK_BUFFER);

			} finally {
				gl.glReadBuffer(current[1]);
				gl.glBindBuffer(GL2.GL_PIXEL_PACK_BUFFER, current[0]);
				if (pbo[0] != 0) gl.glDeleteBuffers(1, pbo, 0);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_2D, 0, 0);
			}

		} finally {
			lock.unlock();
		}
	}

	private void copyPBOToArray(ByteBuffer mappedPBO, Object array) {
		if (array instanceof byte[]) {
			mappedPBO.get((byte[]) array, 0, mappedPBO.capacity());

		} else if (array instanceof short[]) {
			mappedPBO.asShortBuffer().get((short[]) array, 0, mappedPBO.capacity() / 2);

		} else if (array instanceof float[]) {
			mappedPBO.asFloatBuffer().get((float[]) array, 0, mappedPBO.capacity() / 4);

		} else {
			throw new IllegalArgumentException(
					"not array or unsupported array type: "
					+ array.getClass().getName());
		}
	}

	private void clearTexture(Color color) {
		final float a = (float)color.a;
		final float r = (float)color.r * a; 
		final float g = (float)color.g * a; 
		final float b = (float)color.b * a; 

		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();
				gl.glClearColor(r, g, b, a);
				gl.glClear(GL2.GL_COLOR_BUFFER_BIT);
			}
		};

		int pushAttribs = GL2.GL_COLOR_BUFFER_BIT;

		support.useFramebuffer(operation, pushAttribs, this);
	}

	private void clearArray(IArray<?> array, Color color) {
		Object a = array.getArray();
		int len = array.getLength();

		if (color.r==0 && color.g==0 && color.b==0 && color.a==0) {
			if (a instanceof byte[]) {
				Arrays.fill((byte[])a, 0, len, (byte)0);

			} else if (a instanceof short[]) {
				Arrays.fill((short[])a, 0, len, (short)0);

			} else if (a instanceof float[]) {
				Arrays.fill((float[])a, 0, len, 0);

			} else {
				throw new IllegalArgumentException(
						"unsupported array type: " + array.getClass().getName());
			}

		} else {
			// TODO
			throw new UnsupportedOperationException("not implemented");
		}
	}

}
