/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.nio.FloatBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IAccumulationSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class AccumulationSupportImpl implements IAccumulationSupport {

	private final RenderContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram program;

	private final int maxTextureImageUnits;

	@Inject
	public AccumulationSupportImpl(
			RenderContext context, IVideoRenderSupport support,
			IShaderRegistry shaders, GLGlobal glGlobal) {

		this.context = context;
		this.support = support;
		maxTextureImageUnits = glGlobal.getMaxTextureImageUnits();

		String programName = AccumulationSupportImpl.class.getName() + ".ACCUMULATION";
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			program = shaders.registerProgram(
					programName, ShaderType.FRAGMENT_SHADER, null, createAccumulationSource());
		}
		this.program = program;
	}

	public int getMaxSourcesAtATime() {
		return maxTextureImageUnits;
	}

	public void accumulate(List<IVideoBuffer> srcBuffers, List<Double> weights, IVideoBuffer dstBuffer) {
		int numSrc = srcBuffers.size();

		if (numSrc > maxTextureImageUnits) {
			accumulate(srcBuffers.subList(0, maxTextureImageUnits),
						weights.subList(0, maxTextureImageUnits), dstBuffer);
			accumulate(srcBuffers.subList(maxTextureImageUnits, numSrc),
						weights.subList(maxTextureImageUnits, numSrc), dstBuffer);
			return;
		}

		final VideoBounds bounds = dstBuffer.getBounds();

		float[] srcOffsetAndSize = new float[numSrc*4];
		float[] weightArray = new float[numSrc];

		Set<GLUniformData> uniforms = Util.newSet();
		for (int i = 0; i < numSrc; ++i) {
			uniforms.add(new GLUniformData("texSrc"+i, i));

			VideoBounds b = srcBuffers.get(i).getBounds();
			srcOffsetAndSize[i*4  ] = (float)(bounds.x-b.x);
			srcOffsetAndSize[i*4+1] = (float)(bounds.y-b.y);
			srcOffsetAndSize[i*4+2] = b.width;
			srcOffsetAndSize[i*4+3] = b.height;

			weightArray[i] = weights.get(i).floatValue();
		}
		uniforms.add(new GLUniformData("srcOffsetAndSize[0]", 4, FloatBuffer.wrap(srcOffsetAndSize)));
		uniforms.add(new GLUniformData("weights[0]", 1, FloatBuffer.wrap(weightArray)));
		uniforms.add(new GLUniformData("numSrc", numSrc));

		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();
				gl.glEnable(GL2.GL_BLEND);
				gl.glBlendFunc(GL2.GL_ONE, GL2.GL_ONE);
				gl.glBlendEquation(GL2.GL_FUNC_ADD);

				support.ortho2D(bounds);
				support.quad2D(bounds);
			}
		};

		int pushAttribs = GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT;

		support.useShaderProgram(program, uniforms, operation, pushAttribs,
				dstBuffer, srcBuffers.toArray(new IVideoBuffer[srcBuffers.size()]));
	}

	public void accumulate(List<IVideoBuffer> srcBuffers, double weight, IVideoBuffer dstBuffer) {
		accumulate(srcBuffers, Collections.nCopies(srcBuffers.size(), weight), dstBuffer);
	}

	private String[] createAccumulationSource() {
		List<String> source = Util.newList();

		/********************
		uniform sampler2D texSrc0;
		...
		uniform sampler2D texSrcN;			// N = maxTextureImageUnits - 1
		
		uniform vec4 srcOffsetAndSize[M];	// M = maxTextureImageUnits
		********************/

		for (int i = 0; i < maxTextureImageUnits; ++i) {
			source.add(String.format("uniform sampler2D texSrc%d;", i));
		}

		source.add(String.format("uniform vec4 srcOffsetAndSize[%d];", maxTextureImageUnits));

		/********************
		vec4 getSrc(int i) {
			vec4 offAndSize = srcOffsetAndSize[i];
			vec2 coord = (gl_FragCoord.xy + offAndSize.xy) / offAndSize.zw;
			return
				(i == 0) ? texture2D(texSrc0, coord) :
				...
				(i == N) ? texture2D(texSrcN, coord) :
				vec4(0.0);
		}
		********************/

		source.add("vec4 getSrc(int i) {");
		source.add("	vec4 offAndSize = srcOffsetAndSize[i];");
		source.add("	vec2 coord = (gl_FragCoord.xy + offAndSize.xy) / offAndSize.zw;");
		source.add("	return");
		for (int i = 0; i < maxTextureImageUnits; ++i) {
			source.add(String.format("		(i == %1$d) ? texture2D(texSrc%1$d, coord) :", i));
		}
		source.add("		vec4(0.0);");
		source.add("}");


		source.addAll(Arrays.asList(new String[] {
				String.format("uniform float weights[%d];", maxTextureImageUnits),
				"uniform int numSrc;",
				"",
				"void main(void)",
				"{",
				"	vec4 accum = vec4(0.0);",
				"	for (int i = 0; i < numSrc; ++i) {",
				"		accum += weights[i]*getSrc(i);",
				"	}",
				"	gl_FragColor = accum;",
				"}"
		}));

		return source.toArray(new String[source.size()]);
	}

}
