/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.services.IAntiAliasSupport;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class AntiAliasSupportImpl implements IAntiAliasSupport {

	private static final boolean MACOSX = System.getProperty("os.name")
					.toLowerCase().replaceAll("\\s+", "").contains("macosx");

	private final RenderContext context;

	private final int samples;

	private int frameBuffer;

	private final int[] renderBuffer = new int[2];

	private final int[] internalFormat = new int[2];

	private final int[] width = new int[2];

	private final int[] height = new int[2];


	@Inject
	public AntiAliasSupportImpl(RenderContext context, GLGlobal glGlobal) {
		super();
		this.context = context;

		// GL_MAX_SAMPLESが8より大きい場合でも負荷とのバランスを考えて8に制限している。
		// TODO 8より大きいサンプル数も環境設定で指定できるようにする。
		int samples = Math.min(glGlobal.getMaxSamples(), 8);

		// MacでGMA950の場合、サンプル数1で glRenderbufferStorageMultisample を実行しても、
		// glBlitFramebuffer でクラッシュしてしまう。
		// また、WindowsでGMA X4500の場合も glFramebufferRenderbuffer 後にエラーとなる。
		// そのため、サンプル数1の場合は別処理にする。
		this.samples = samples > 0 ? samples : 0;
	}

	void dispose() {
		GL2 gl = context.getGL().getGL2();

		for (int i = 0; i < 2; ++i) {
			internalFormat[i] = 0;
			width[i] = 0;
			height[i] = 0;

			if (renderBuffer[i] != 0) {
				gl.glDeleteRenderbuffers(1, new int[] { renderBuffer[i] }, 0);
				renderBuffer[i] = 0;
			}
		}

		if (frameBuffer != 0) {
			gl.glDeleteFramebuffers(1, new int[] { frameBuffer }, 0);
			frameBuffer = 0;
		}
	}

	private int getFrameBuffer(GL2 gl) {
		if (frameBuffer == 0) {
			int[] fb = new int[1];
			gl.glGenFramebuffers(1, fb, 0);
			frameBuffer = fb[0];
		}
		return frameBuffer;
	}

	private int getRenderBuffer(GL2 gl, int i, int internalFormat, int width, int height) {
		if (renderBuffer[i] == 0) {
			int[] rb = new int[1];
			gl.glGenRenderbuffers(1, rb, 0);
			renderBuffer[i] = rb[0];
		}

		if (internalFormat != this.internalFormat[i] || width != this.width[i] || height != this.height[i]) {
			gl.glBindRenderbuffer(GL2.GL_RENDERBUFFER, renderBuffer[i]);
			gl.glRenderbufferStorageMultisample(GL2.GL_RENDERBUFFER, samples, internalFormat, width, height);
			this.internalFormat[i] = internalFormat;
			this.width[i] = width;
			this.height[i] = height;
		}

		return renderBuffer[i];
	}

	private int getColorRenderBuffer(GL2 gl, int width, int height) {
		// TODO あまりに大きなサイズの場合の対策が必要。
		//      例えば、過去数十回程度の使用履歴を記録しておいて、その中で一番大きなサイズまで縮小するとか、
		//      もっと単純に、数十回使用する毎に小さなサイズに作りなおす等。
		width = Math.max(width, this.width[0]);
		height = Math.max(height, this.height[0]);

		return getRenderBuffer(gl, 0, context.getColorMode().glInternalFormat, width, height);
	}

	private int getDepthRenderBuffer(GL2 gl) {
		// TODO 常時 GL_DEPTH_COMPONENT32 を指定した方が良いかも？
		int internalFormat = (context.getColorMode() == ColorMode.RGBA32_FLOAT)
								? GL2.GL_DEPTH_COMPONENT32 : GL2.GL_DEPTH_COMPONENT;
		return getRenderBuffer(gl, 1, internalFormat, width[0], height[0]);
	}

	private void clear(GL2 gl, boolean clearDepth, Color clearColor) {
		int bits = 0;

		if (clearDepth) {
			gl.glClearDepth(1.0);
			bits |= GL2.GL_DEPTH_BUFFER_BIT;
		}

		if (clearColor != null) {
			float a = (float)clearColor.a;
			float r = (float)clearColor.r * a;
			float g = (float)clearColor.g * a;
			float b = (float)clearColor.b * a;
			gl.glClearColor(r, g, b, a);
			bits |= GL2.GL_COLOR_BUFFER_BIT;
		}

		if (bits != 0) {
			gl.glClear(bits);
		}
	}

	public void antiAlias(int width, int height, boolean depthTest, Color clearColor, Runnable operation) {
		if (width <= 0 || height <= 0) {
			// TODO 例外を投げた方がいいか？
			//throw new IllegalArgumentException();
			return;
		}

		if (samples == 0) {
			runWithoutRenderbuffer(width, height, depthTest, clearColor, operation);
			return;
		}

		GL2 gl = context.getGL().getGL2();

		int[] current = new int[3];
		gl.glGetIntegerv(GL2.GL_DRAW_FRAMEBUFFER_BINDING, current, 0);
		gl.glGetIntegerv(GL2.GL_READ_FRAMEBUFFER_BINDING, current, 1);
		gl.glGetIntegerv(GL2.GL_RENDERBUFFER_BINDING, current, 2);
		try {
			int fb = getFrameBuffer(gl);
			gl.glBindFramebuffer(GL2.GL_DRAW_FRAMEBUFFER, fb);

			gl.glPushAttrib(GL2.GL_COLOR_BUFFER_BIT | (depthTest ? GL2.GL_ENABLE_BIT | GL2.GL_DEPTH_BUFFER_BIT : 0));
			try {
				gl.glFramebufferRenderbuffer(GL2.GL_DRAW_FRAMEBUFFER, GL2.GL_COLOR_ATTACHMENT0,
						GL2.GL_RENDERBUFFER, getColorRenderBuffer(gl, width, height));
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				if (depthTest) {
					gl.glFramebufferRenderbuffer(GL2.GL_DRAW_FRAMEBUFFER, GL2.GL_DEPTH_ATTACHMENT,
							GL2.GL_RENDERBUFFER, getDepthRenderBuffer(gl));
					gl.glEnable(GL2.GL_DEPTH_TEST);

				} else if (MACOSX && renderBuffer[1] != 0) {
					// Macの場合、一度デプスバッファ用のレンダーバッファをアタッチすると
					// それ以降、カラーバッファ用のレンダーバッファとデプスバッファ用の
					// レンダーバッファのサイズが常に一致していないと正常に動作しない。
					// そのため、一度でもデプステストを使用した後はデプスバッファ用の
					// レンダーバッファを常に設定する。
					gl.glFramebufferRenderbuffer(GL2.GL_DRAW_FRAMEBUFFER, GL2.GL_DEPTH_ATTACHMENT,
							GL2.GL_RENDERBUFFER, getDepthRenderBuffer(gl));
				}

				clear(gl, depthTest, clearColor);
				operation.run();

				gl.glBindFramebuffer(GL2.GL_READ_FRAMEBUFFER, fb);
				gl.glBindFramebuffer(GL2.GL_DRAW_FRAMEBUFFER, current[0]);
				gl.glBlitFramebuffer(0, 0, width, height, 0, 0, width, height, GL2.GL_COLOR_BUFFER_BIT, GL2.GL_NEAREST);

			} finally {
				gl.glFramebufferRenderbuffer(GL2.GL_DRAW_FRAMEBUFFER, GL2.GL_DEPTH_ATTACHMENT, GL2.GL_RENDERBUFFER, 0);
				gl.glFramebufferRenderbuffer(GL2.GL_DRAW_FRAMEBUFFER, GL2.GL_COLOR_ATTACHMENT0, GL2.GL_RENDERBUFFER, 0);
				gl.glPopAttrib();
			}
		} finally {
			gl.glBindRenderbuffer(GL2.GL_RENDERBUFFER, current[2]);
			gl.glBindFramebuffer(GL2.GL_READ_FRAMEBUFFER, current[1]);
			gl.glBindFramebuffer(GL2.GL_DRAW_FRAMEBUFFER, current[0]);
		}
	}

	public void antiAlias(int width, int height, Runnable operation) {
		antiAlias(width, height, false, Color.COLORLESS_TRANSPARENT, operation);
	}

	private void runWithoutRenderbuffer(int width, int height, boolean depthTest, Color clearColor, Runnable operation) {
		GL2 gl = context.getGL().getGL2();

		int depthTexture = 0;
		gl.glPushAttrib(GL2.GL_COLOR_BUFFER_BIT | (depthTest ? GL2.GL_ENABLE_BIT | GL2.GL_DEPTH_BUFFER_BIT : 0));
		try {
			if (depthTest) {
				depthTexture = createDepthTexture(gl, width, height);
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_DEPTH_ATTACHMENT, GL2.GL_TEXTURE_2D, depthTexture, 0);
				gl.glEnable(GL2.GL_DEPTH_TEST);
			}

			clear(gl, depthTest, clearColor);
			operation.run();

		} finally {
			if (depthTest) {
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_DEPTH_ATTACHMENT, GL2.GL_TEXTURE_2D, 0, 0);
				if (depthTexture != 0) gl.glDeleteTextures(1, new int[] { depthTexture }, 0);
			}
			gl.glPopAttrib();
		}
	}

	private static final float[] FLOAT0000 = new float[] { 0, 0, 0, 0 };

	private int createDepthTexture(GL2 gl, int width, int height) {
		// TODO 常時 GL_DEPTH_COMPONENT32 を指定した方が良いかも？
		int internalFormat = (context.getColorMode() == ColorMode.RGBA32_FLOAT)
								? GL2.GL_DEPTH_COMPONENT32 : GL2.GL_DEPTH_COMPONENT;

		int[] texture = new int[1];
		int[] current = new int[1];
		gl.glGetIntegerv(GL2.GL_TEXTURE_BINDING_2D, current, 0);
		try {
			gl.glGenTextures(1, texture, 0);
			gl.glBindTexture(GL2.GL_TEXTURE_2D, texture[0]);

			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_NEAREST);
			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_NEAREST);
			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_BORDER);
			gl.glTexParameteri(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_BORDER);
			gl.glTexParameterfv(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_BORDER_COLOR, FLOAT0000, 0);
			gl.glTexImage2D(GL2.GL_TEXTURE_2D, 0, internalFormat,
					Math.max(1, width), Math.max(1, height), 0, GL2.GL_DEPTH_COMPONENT, GL2.GL_FLOAT, null);

			int result = texture[0];
			texture[0] = 0;
			return result;

		} finally {
			gl.glBindTexture(GL2.GL_TEXTURE_2D, current[0]);
			if (texture[0] != 0) gl.glDeleteTextures(1, texture, 0);
		}
	}

	public int getSamples() {
		return samples;
	}

}
