/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.lang.reflect.Array;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ArrayPool<T> {

	private static final Logger _logger = LoggerFactory.getLogger(ArrayPool.class);


	private static final int MAX_QUEUE_SIZE = 16;

	private static final int[] ARRAY_LENGTHS = { 4096, 16384, 65536, 262144, 524288, 1048576, 2097152, 4194304, 8388608, 16777216 };


	private class PoolQueue {

		private final int _length;

		private final ConcurrentLinkedQueue<T> _queue = new ConcurrentLinkedQueue<T>();

		private final AtomicInteger _count = new AtomicInteger();


		private PoolQueue(int length) {
			_length = length;
		}

		private T get() {
			T array = _queue.poll();
			if (array != null) {
				_count.decrementAndGet();
			} else {
				array = newArray(_length);
				_logger.info(String.format(
						"new array created: %s[%d]", _componentType.getName(), _length));
			}
			return array;
		}

		private void put(T array) {
			if (_count.get() < MAX_QUEUE_SIZE) {
				_queue.add(array);
				_count.incrementAndGet();
			} else {
				_logger.info(String.format(
						"queue is reached to max size: %s[%d]", _componentType.getName(), _length));
			}
		}
	}


	private final Class<?> _componentType;

	private final ConcurrentHashMap<Integer, PoolQueue> _map = new ConcurrentHashMap<Integer, PoolQueue>();


	ArrayPool(Class<?> componentType) {
		super();
		_componentType = componentType;
	}

	@SuppressWarnings("unchecked")
	private T newArray(int length) {
		return (T) Array.newInstance(_componentType, length);
	}

	private PoolQueue getQueue(int length) {
		Integer key = length;
		if (!_map.containsKey(key)) {
			_map.putIfAbsent(key, new PoolQueue(length));
		}
		return _map.get(key);
	}

	T get(int length) {
		for (int i = 0; i < ARRAY_LENGTHS.length; ++i) {
			int n = ARRAY_LENGTHS[i];
			if (length <= n) {
				return getQueue(n).get();
			}
		}
		_logger.warn(String.format("length is too big to pool: %s[%d]", _componentType.getName(), length));
		return newArray(length);
	}

	void put(T array) {
		if (array == null) {
			return;
		}

		int length = Array.getLength(array);
		for (int i = 0; i < ARRAY_LENGTHS.length; ++i) {
			if (length == ARRAY_LENGTHS[i]) {
				getQueue(length).put(array);
				return;
			}
		}

		// 最大サイズを超えている場合はgetの際に警告を出しているので、何もせず無視。
		if (length <= ARRAY_LENGTHS[ARRAY_LENGTHS.length-1]) {
			throw new IllegalArgumentException("array length does not match: " + length);
		}
	}

}
