/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.blurSharpen;

import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IBlurSupport.BlurDimensions;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.UnsharpMask", category=Categories.BLUR_AND_SHARPEN)
public class UnsharpMask {

	@Property(value="50", min="0", max="1000")
	private IAnimatableDouble amount;

	@Property(value="1", min="0", max="500")
	private IAnimatableDouble radius;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble threshold;

	@Property("true")
	private IAnimatableBoolean fast;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlurSupport blurSupport;

	private final IShaderProgram programNoThreshold;

	private final IShaderProgram programWithThreshold;

	@Inject
	public UnsharpMask(
			IVideoEffectContext context, IVideoRenderSupport support,
			IBlurSupport blurSupport, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blurSupport = blurSupport;

		programNoThreshold = shaders.getProgram(UnsharpMask.class, "UNSHARP_MASK");
		programWithThreshold = shaders.getProgram(UnsharpMask.class, "UNSHARP_MASK_T");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		if (source.getBounds().isEmpty()) {
			return source;
		}

		double amount = context.value(this.amount) / 100;
		if (amount == 0) {
			return source;
		}

		Resolution resolution = context.getVideoResolution();
		double radius = resolution.scale(context.value(this.radius));
		if (radius == 0) {
			return source;
		}

		double threshold = context.value(this.threshold) / 100;
		if (threshold == 1) {
			return source;
		}

		boolean fast = context.value(this.fast);


		IVideoBuffer blur = null;
		try {
			blur = blurSupport.gaussianBlur(source, radius, BlurDimensions.BOTH, true, fast);

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("srcTex", 0));
			uniforms.add(new GLUniformData("blrTex", 1));
			uniforms.add(new GLUniformData("amount", (float)amount));

			IShaderProgram program;
			if (threshold == 0) {
				program = programNoThreshold;
			} else {
				program = programWithThreshold;
				uniforms.add(new GLUniformData("threshold", (float)(2*threshold-1)));
			}

			return support.useShaderProgram(program, uniforms, null, source, blur);

		} finally {
			if (blur != null) blur.dispose();
			source.dispose();
		}
	}

	private static String[] createProgramSource(boolean withThreshold) {
		boolean t = withThreshold;
		return new String[] {
			t ? "#define WITH_THRESHOLD" : "",
				"",
				"uniform sampler2D srcTex;",
				"uniform sampler2D blrTex;",
				"uniform float amount;",
				"",
				"#ifdef WITH_THRESHOLD",
				"	uniform float threshold;",
				"#endif",
				"",
				"void main(void)",
				"{",
				"	vec2 tc = gl_TexCoord[0].st;",
				"	vec4 srcColor = texture2D(srcTex, tc);",
				"	vec4 blrColor = texture2D(blrTex, tc);",
				"",
				"	srcColor.rgb = (srcColor.a > 0.0) ? srcColor.rgb/srcColor.a : vec3(0.0);",
				"	blrColor.rgb = (blrColor.a > 0.0) ? blrColor.rgb/blrColor.a : vec3(0.0);",
				"",
				"	vec3 diff = srcColor.rgb - blrColor.rgb;",
				"",
				"#ifdef WITH_THRESHOLD",
				"	diff *= clamp(abs(diff)-vec3(threshold), 0.0, 1.0);",
				"#endif",
				"",
				"	gl_FragColor = vec4((srcColor.rgb+diff*amount)*srcColor.a, srcColor.a);",
				"}"
		};
	}

	@ShaderSource
	public static final String[] UNSHARP_MASK = createProgramSource(false);

	@ShaderSource
	public static final String[] UNSHARP_MASK_T = createProgramSource(true);

}
