/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.stylize;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Quality;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.MotionTile", category=Categories.STYLIZE)
public class MotionTile {

	@Property
	private IAnimatableVec2d tileCenter;

	@Property(value="100", min="0", max="100")
	private IAnimatableVec2d tileSize;

	@Property(value="100", min="0"/*, max="1000"*/)
	private IAnimatableVec2d outputSize;

	@Property
	private IAnimatableBoolean mirrorEdges;

	@Property
	private IAnimatableDouble phase;

	@Property
	private IAnimatableBoolean horizontalPhaseShift;

	@Property("true")
	private IAnimatableBoolean interpolation;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram hPhaseProgram;

	private final IShaderProgram vPhaseProgram;

	@Inject
	public MotionTile(IVideoEffectContext context,
			IVideoRenderSupport support, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		hPhaseProgram = shaders.getProgram(MotionTile.class, "HPHASE");
		vPhaseProgram = shaders.getProgram(MotionTile.class, "VPHASE");
	}

	private VideoBounds calcOutputBounds(
			VideoBounds sourceBounds, Vec2d outputSize, double[] ltrb) {

		double w = sourceBounds.width * outputSize.x / 100;
		double h = sourceBounds.height * outputSize.y / 100;

		double left   = sourceBounds.x - (w - sourceBounds.width) / 2;
		double top    = sourceBounds.y - (h - sourceBounds.height) / 2;
		double right  = left + w;
		double bottom = top + h;

		if (ltrb != null) {
			ltrb[0] = left;
			ltrb[1] = top;
			ltrb[2] = right;
			ltrb[3] = bottom;
		}

		double x = sourceBounds.x + (int)Math.floor(left - sourceBounds.x);
		double y = sourceBounds.y + (int)Math.floor(top - sourceBounds.y);
		return new VideoBounds(x, y, (int)Math.ceil(right-x), (int)Math.ceil(bottom-y));
	}

	public VideoBounds getVideoBounds() {
		VideoBounds bounds = context.getPreviousBounds();
		if (bounds.isEmpty()) {
			return bounds;
		}

		Vec2d outputSize = context.value(this.outputSize);
		return calcOutputBounds(bounds, outputSize, null);
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		VideoBounds bounds = source.getBounds();
		if (bounds.isEmpty()) {
			return source;
		}

		IVideoBuffer buf1 = null;
		IVideoBuffer buf2 = null;
		try {
			Resolution resolution = context.getVideoResolution();
			Quality quality = context.getQuality();

			Vec2d tileCenter = resolution.scale(context.value(this.tileCenter));
			Vec2d tileSize = context.value(this.tileSize);
			Vec2d outputSize = context.value(this.outputSize);
			boolean mirror = context.value(this.mirrorEdges);
			double phase = context.value(this.phase) / 360;
			boolean hPhase = context.value(this.horizontalPhaseShift);
			boolean interpolation = context.value(this.interpolation);


			double tileWidth = Math.max(1, bounds.width * tileSize.x / 100);
			double tileHeight = Math.max(1, bounds.height * tileSize.y / 100);

			final VideoBounds tileBounds = new VideoBounds((int)tileWidth, (int)tileHeight);
			buf1 = support.createVideoBuffer(tileBounds);

			source.setTextureFilter(quality == Quality.DRAFT || resolution.scale < 1 ? TextureFilter.NEAREST
									: interpolation ? TextureFilter.MIPMAP : TextureFilter.LINEAR);
			source.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();
					gl.glColor4f(1, 1, 1, 1);
					support.ortho2D(tileBounds);
					support.quad2D(tileBounds, new double[][] {{0,0},{1,0},{1,1},{0,1}});
				}
			};
			int pushAttribs = GL2.GL_CURRENT_BIT;
			support.useFramebuffer(operation, pushAttribs, buf1, source);

			source.dispose();
			source = null;


			double[] ltrb = new double[4];
			VideoBounds outputBounds = calcOutputBounds(bounds, outputSize, ltrb);
			buf2 = support.createVideoBuffer(outputBounds);

			ltrb[0] -= outputBounds.x;
			ltrb[1] -= outputBounds.y;
			ltrb[2] -= outputBounds.x;
			ltrb[3] -= outputBounds.y;

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("tileCenter", 2, toFloatBuffer(
					tileCenter.x-outputBounds.x, tileCenter.y-outputBounds.y)));
			uniforms.add(new GLUniformData("tileSize", 2, toFloatBuffer(tileWidth, tileHeight)));
			uniforms.add(new GLUniformData("phase", (float)phase));
			uniforms.add(new GLUniformData("ltrb", 4, toFloatBuffer(ltrb)));

			buf1.setTextureWrapMode(mirror
								? TextureWrapMode.MIRRORED_REPEAT : TextureWrapMode.REPEAT);
			buf1.setTextureFilter(quality == Quality.DRAFT || resolution.scale < 1
								? TextureFilter.NEAREST : TextureFilter.LINEAR);

			IShaderProgram program = hPhase ? hPhaseProgram : vPhaseProgram;
			support.useShaderProgram(program, uniforms, buf2, buf1);

			IVideoBuffer result = buf2;
			buf2 = null;
			return result;

		} finally {
			if (source != null) source.dispose();
			if (buf1 != null) buf1.dispose();
			if (buf2 != null) buf2.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double...values) {
		float[] farray = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			farray[i] = (float)values[i];
		}
		return FloatBuffer.wrap(farray);
	}


	@ShaderSource
	public static final String[] HPHASE = createProgramSource(true);

	@ShaderSource
	public static final String[] VPHASE = createProgramSource(false);


	private static String[] createProgramSource(boolean hPhase) {
		return new String[] {
	   hPhase ?	"#define HPHASE" : "",
				"",
				"uniform sampler2D texture;",
				"uniform vec2 tileCenter;",
				"uniform vec2 tileSize;",
				"uniform float phase;",
				"uniform vec4 ltrb;",
				"",
				"void main(void)",
				"{",
				"	vec2 xy = gl_FragCoord.xy;",
				"	vec2 tc = (xy-tileCenter)/tileSize + vec2(0.5);",
				"#ifdef HPHASE",
				"	tc.x -= step(mod(tc.y, 2.0), 1.0) * phase;",
				"#else",
				"	tc.y -= step(mod(tc.x, 2.0), 1.0) * phase;",
				"#endif",
				"	vec4 a = clamp(vec4(xy-ltrb.xy, ltrb.zw-xy)+vec4(0.5), 0.0, 1.0);",
				"	a.xy *= a.zw;",
				"	gl_FragColor = texture2D(texture, tc)*a.x*a.y;",
				"}"
		};
	}

}
