/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.examples.effect;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IObjectArray;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.services.IAudioEffectContext;

import com.google.inject.Inject;

/*
 * オーディオエフェクトのサンプル。
 * 音量を調整します。
 */

/*
 * plugin.xml の Extensions タブで追加したカテゴリ（“カテゴリ追加のサンプル”というカテゴリ名）に分類しています。
 */
@Effect(id="ch.kuramo.javie.examples.effect.AudioLevel",
		category="ch.kuramo.javie.examples.effect.exampleCategory")
public class AudioLevel {

	@Property(value="100", min="0", max="400")
	private IAnimatableDouble level;


	private final IAudioEffectContext context;

	@Inject
	public AudioLevel(IAudioEffectContext context) {
		this.context = context;
	}

	/*
	 * オーディオフェクトクラスには、doAudioEffect という名前で
	 * 戻り値が IAudioBuffer の public なメソッドを実装する必要があります。
	 */
	public IAudioBuffer doAudioEffect() {
		IAudioBuffer buffer = null;
		IObjectArray<Double> levels = null;

		try {
			// ひとつ前のエフェクトまで実行し、結果のオーディオデータを受け取ります。
			// これには context.getAudioFrameCount() が示すサンプルフレーム数のデータを含んでいます。
			// (ここでのフレームとビデオのフレームは無関係です)
			buffer = context.doPreviousEffect();

			// context.doPreviousEffect() を呼び出す前に context.setAudioFrameCount() を呼び出すことで、
			// サンプルフレーム数を変更することができます。サンプルフレーム数を変更する必要があるのは、
			// 現在時刻のエフェクトの処理に、過去または未来のデータを必要とする場合等です。
			// その場合、context.setTime() とともに context.setAudioFrameCount() を用いて、
			// 必要な時間範囲のオーディオデータを取得してください。
			// また、context.setTime(), context.setAudioFrameCount() 及び context.doPreviousEffect() は
			// 繰り返し呼び出すこともできます。
			//
			// サンプルフレーム数を変更した場合でも、doAudioEffect メソッドから返す IAudioBuffer オブジェクトは
			// 元のサンプルフレーム数にちょうど一致する量のバッファである必要があることに注意してください。
			// したがって、サンプルフレーム数を変更する場合は最初に(サンプルフレーム数を変更する前に)、
			// 
			//   IAudioBuffer newBuffer = context.createAudioBuffer();
			//
			// として、元のサンプルフレーム数にちょうど一致する IAudioBuffer オブジェクトを新たに作成し、
			// エフェクトの処理結果をこのバッファにコピーして doAudioEffect メソッドから返す必要があります。
			//
			//
			// 一方、サンプルフレーム数を変更しない場合は、context.doPreviousEffect() から得たバッファに
			// 処理結果を上書きすることができます。以下に示すサンプルはこちらのケースです。


			// プロパティの値を取得します。
			// これは、context.getAudioFrameCount() の数に一致するサイズの配列です。
			// プロパティをアニメーションさせていない場合は全て同じ値となりますが、
			// アニメーションさせている場合は context.doPreviousEffect() から得た
			// 各サンプルフレームの時刻に対応する値となります。
			levels = context.values(level);


			// buffer.getData() メソッドが返す配列はプールされた配列を再利用しているため実際のデータ量以上の長さとなっています。
			// そのため、配列のlength属性による長さを使用してはいけません。
			// 必ず buffer.getFrameCount() または buffer.getDataLength() が返す値を使用してください。
			Object data = buffer.getData();
			int frameCount = buffer.getFrameCount();


			// 16bit整数、32bit整数、32bit浮動小数のそれぞれに対応した処理を行う必要があります。
			switch (buffer.getAudioMode().dataType) {
				case SHORT:
					audioLevel((short[]) data, frameCount, levels);
					break;
	
				case INT:
					audioLevel((int[]) data, frameCount, levels);
					break;
	
				case FLOAT:
					audioLevel((float[]) data, frameCount, levels);
					break;
			}

			IAudioBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
			if (levels != null) levels.release(); 
		}
	}

	private void audioLevel(short[] data, int frameCount, IObjectArray<Double> levels) {
		Object[] levelsArray = levels.getArray();

		for (int i = 0; i < frameCount; ++i) {
			double level = (Double)levelsArray[i] / 100.0;

			data[i*2  ] = clampToShort(data[i*2  ] * level);
			data[i*2+1] = clampToShort(data[i*2+1] * level);
		}
	}

	private void audioLevel(int[] data, int frameCount, IObjectArray<Double> levels) {
		Object[] levelsArray = levels.getArray();

		for (int i = 0; i < frameCount; ++i) {
			double level = (Double)levelsArray[i] / 100.0;

			data[i*2  ] = clampToInt(data[i*2  ] * level);
			data[i*2+1] = clampToInt(data[i*2+1] * level);
		}
	}

	private void audioLevel(float[] data, int frameCount, IObjectArray<Double> levels) {
		Object[] levelsArray = levels.getArray();

		for (int i = 0; i < frameCount; ++i) {
			double level = (Double)levelsArray[i] / 100.0;

			data[i*2  ] = (float)(data[i*2  ] * level);
			data[i*2+1] = (float)(data[i*2+1] * level);
		}
	}

	private short clampToShort(double d) {
		return (short) Math.min(Math.max(d, Short.MIN_VALUE), Short.MAX_VALUE); 
	}

	private int clampToInt(double d) {
		return (int) Math.min(Math.max(d, Integer.MIN_VALUE), Integer.MAX_VALUE); 
	}

}
