/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ftgl;

import javax.media.opengl.glu.GLU;
import javax.media.opengl.glu.GLUtessellator;
import javax.media.opengl.glu.GLUtessellatorCallback;
import javax.media.opengl.glu.GLUtessellatorCallbackAdapter;

import com.sun.jna.Callback;
import com.sun.jna.Pointer;

public interface FTVectoriserMakeMeshCallback extends Callback {

	static final FTVectoriserMakeMeshCallback CALLBACK = new FTVectoriserMakeMeshCallbackImpl();

	void callback(
			Pointer contourList, int contourCount, int contourFlag,
			Pointer meshPtr, double zNormal, int outsetType, float outsetSize);

}

class FTVectoriserMakeMeshCallbackImpl implements FTVectoriserMakeMeshCallback {

	private static final int ft_outline_even_odd_fill = 0x2;

	private static final GLUtessellatorCallback tessCallback = new GLUtessellatorCallbackAdapter() {

		public void errorData(int errnum, Object mesh) {
			((FTMesh) mesh).Error(errnum);
		}

		public void vertexData(Object vertexData, Object mesh) {
			double[] vertex = (double[]) vertexData;
			((FTMesh) mesh).AddPoint(vertex[0], vertex[1], vertex[2]);
		}

		public void combineData(
				double[] coords, Object[] data, float[] weight, Object[] outData, Object mesh) {

			outData[0] = ((FTMesh) mesh).Combine(coords[0], coords[1], coords[2]);
		}

		public void beginData(int type, Object mesh) {
			((FTMesh) mesh).Begin(type);
		}

		public void endData(Object mesh) {
			((FTMesh) mesh).End();
		}

	};

	public void callback(
			Pointer contourList, int contourCount, int contourFlag,
			Pointer meshPtr, double zNormal, int outsetType, float outsetSize) {

		GLUtessellator tobj = GLU.gluNewTess();
		try {

			GLU.gluTessCallback(tobj, GLU.GLU_TESS_BEGIN_DATA, tessCallback);
			GLU.gluTessCallback(tobj, GLU.GLU_TESS_VERTEX_DATA, tessCallback);
			GLU.gluTessCallback(tobj, GLU.GLU_TESS_COMBINE_DATA, tessCallback);
			GLU.gluTessCallback(tobj, GLU.GLU_TESS_END_DATA, tessCallback);
			GLU.gluTessCallback(tobj, GLU.GLU_TESS_ERROR_DATA, tessCallback);

			if ((contourFlag & ft_outline_even_odd_fill) != 0) {
				GLU.gluTessProperty(tobj, GLU.GLU_TESS_WINDING_RULE, GLU.GLU_TESS_WINDING_ODD);
			} else {
				GLU.gluTessProperty(tobj, GLU.GLU_TESS_WINDING_RULE, GLU.GLU_TESS_WINDING_NONZERO);
			}

			GLU.gluTessProperty(tobj, GLU.GLU_TESS_TOLERANCE, 0);
			GLU.gluTessNormal(tobj, 0.0f, 0.0f, zNormal);
			GLU.gluTessBeginPolygon(tobj, new FTMesh(meshPtr));

			FTContour contour = new FTContour();
			for (Pointer ptr : contourList.getPointerArray(0, contourCount)) {
				contour.setPointer(ptr);
				switch(outsetType) {
					case 1: contour.buildFrontOutset(outsetSize); break;
					case 2: contour.buildBackOutset(outsetSize); break;
				}

				GLU.gluTessBeginContour(tobj);

				for(int p = 0, n = contour.PointCount(); p < n; ++p) {
					double[] d;
					switch (outsetType) {
						case 1: d = contour.FrontPoint(p); break;
						case 2: d = contour.BackPoint(p); break;
						case 0: default: d = contour.Point(p); break;
					}
					GLU.gluTessVertex(tobj, d, 0, d);
				}

				GLU.gluTessEndContour(tobj);
			}

			GLU.gluTessEndPolygon(tobj);

		} finally {
			GLU.gluDeleteTess(tobj);
		}
	}

}

class FTContour {

	private Pointer p;

	void setPointer(Pointer p) {
		this.p = p;
	}

	double[] Point(int index) {
		double[] point = new double[3];
		FTGL.ftContour_Point(p, index, point);
		return point;
	}

	double[] FrontPoint(int index) {
		double[] point = new double[3];
		FTGL.ftContour_FrontPoint(p, index, point);
		return point;
	}

	double[] BackPoint(int index) {
		double[] point = new double[3];
		FTGL.ftContour_BackPoint(p, index, point);
		return point;
	}

	void buildFrontOutset(float outset) {
		FTGL.ftContour_buildFrontOutset(p, outset);
	}

	void buildBackOutset(float outset) {
		FTGL.ftContour_buildBackOutset(p, outset);
	}

	int PointCount() {
		return FTGL.ftContour_PointCount(p);
	}

}

class FTMesh {

	private final Pointer p;

	FTMesh(Pointer p) {
		this.p = p;
	}

	void AddPoint(double x, double y, double z) {
		FTGL.ftMesh_AddPoint(p, x, y, z);
	}

	double[] Combine(double x, double y, double z) {
		double[] point = new double[3];
		FTGL.ftMesh_Combine(p, x, y, z, point);
		return point;
	}

	void Begin(int meshType) {
		FTGL.ftMesh_Begin(p, meshType);
	}

	void End() {
		FTGL.ftMesh_End(p);
	}

	void Error(int e) {
		FTGL.ftMesh_Error(p, e);
	}

}
