/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import noise.ImprovedNoise;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.core.AnimatableBoolean;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableInteger;
import ch.kuramo.javie.core.CoreContext;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.TAWigglySelector;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

@ProjectElement("TAWigglySelector")
public class TAWigglySelectorImpl extends AbstractTASelector implements TAWigglySelector {

	private AnimatableMode mode = new AnimatableMode(Mode.INTERSECT);

	private AnimatableDouble maxAmount = new AnimatableDouble(100d, -100d, 100d);

	private AnimatableDouble minAmount = new AnimatableDouble(-100d, -100d, 100d);

	private AnimatableDouble wigglesPerSecond = new AnimatableDouble(2d);

	private AnimatableDouble correlation = new AnimatableDouble(50d, 0d, 100d);

	private AnimatableDouble temporalPhase = new AnimatableDouble(0d);

	private AnimatableDouble spatialPhase = new AnimatableDouble(0d);

	private AnimatableBoolean lockDimensions = new AnimatableBoolean(Boolean.FALSE);

	private AnimatableInteger randomSeed = new AnimatableInteger(0, 0, 10000);


	private final RenderContext context;

	@Inject
	public TAWigglySelectorImpl(RenderContext context) {
		super("ウィグリーセレクタ");
		this.context = context;
	}

	public AnimatableMode getMode() {
		return mode;
	}

	public void setMode(AnimatableMode mode) {
		mode.copyConfigurationFrom(this.mode);
		this.mode = mode;
	}

	public AnimatableDouble getMaxAmount() {
		return maxAmount;
	}

	public void setMaxAmount(AnimatableDouble maxAmount) {
		maxAmount.copyConfigurationFrom(this.maxAmount);
		this.maxAmount = maxAmount;
	}

	public AnimatableDouble getMinAmount() {
		return minAmount;
	}

	public void setMinAmount(AnimatableDouble minAmount) {
		minAmount.copyConfigurationFrom(this.minAmount);
		this.minAmount = minAmount;
	}

	public AnimatableDouble getWigglesPerSecond() {
		return wigglesPerSecond;
	}

	public void setWigglesPerSecond(AnimatableDouble wigglesPerSecond) {
		wigglesPerSecond.copyConfigurationFrom(this.wigglesPerSecond);
		this.wigglesPerSecond = wigglesPerSecond;
	}

	public AnimatableDouble getCorrelation() {
		return correlation;
	}

	public void setCorrelation(AnimatableDouble correlation) {
		correlation.copyConfigurationFrom(this.correlation);
		this.correlation = correlation;
	}

	public AnimatableDouble getTemporalPhase() {
		return temporalPhase;
	}

	public void setTemporalPhase(AnimatableDouble temporalPhase) {
		temporalPhase.copyConfigurationFrom(this.temporalPhase);
		this.temporalPhase = temporalPhase;
	}

	public AnimatableDouble getSpatialPhase() {
		return spatialPhase;
	}

	public void setSpatialPhase(AnimatableDouble spatialPhase) {
		spatialPhase.copyConfigurationFrom(this.spatialPhase);
		this.spatialPhase = spatialPhase;
	}

	public AnimatableBoolean getLockDimensions() {
		return lockDimensions;
	}

	public void setLockDimensions(AnimatableBoolean lockDimensions) {
		lockDimensions.copyConfigurationFrom(this.lockDimensions);
		this.lockDimensions = lockDimensions;
	}

	public AnimatableInteger getRandomSeed() {
		return randomSeed;
	}

	public void setRandomSeed(AnimatableInteger randomSeed) {
		randomSeed.copyConfigurationFrom(this.randomSeed);
		this.randomSeed = randomSeed;
	}

	public void prepareExpression(ExpressionScope scope) {
		scope.assignTo(mode);
		scope.assignTo(maxAmount);
		scope.assignTo(minAmount);
		scope.assignTo(wigglesPerSecond);
		scope.assignTo(correlation);
		scope.assignTo(temporalPhase);
		scope.assignTo(spatialPhase);
		scope.assignTo(lockDimensions);
		scope.assignTo(randomSeed);
	}

	public Object createExpressionElement(CoreContext context) {
		return new TAWigglySelectorExpressionElement(context);
	}

	public class TAWigglySelectorExpressionElement extends TASelectorExpressionElement {

		public TAWigglySelectorExpressionElement(CoreContext context) {
			super(context);
		}

		public Object getMode()				{ return elem(mode); }
		public Object getMaxAmount()		{ return elem(maxAmount); }
		public Object getMinAmount()		{ return elem(minAmount); }
		public Object getWigglesPerSecond()	{ return elem(wigglesPerSecond); }
		public Object getCorrelation()		{ return elem(correlation); }
		public Object getTemporalPhase()	{ return elem(temporalPhase); }
		public Object getSpatialPhase()		{ return elem(spatialPhase); }
		public Object getLockDimensions()	{ return elem(lockDimensions); }
		public Object getRandomSeed()		{ return elem(randomSeed); }
	}

	public Evaluator createEvaluator(int[] totals, Time mediaTime) {
		final Mode mode = this.mode.value(context);
		final double maxAmount = this.maxAmount.value(context) * 0.01;
		final double minAmount = this.minAmount.value(context) * 0.01;
		double wigglesPerSecond = this.wigglesPerSecond.value(context);
		final double correlation = this.correlation.value(context) * 0.01;
		double temporalPhase = this.temporalPhase.value(context) / 360;
		final double spatialPhase = this.spatialPhase.value(context) / 360;
		final boolean lockDimensions = this.lockDimensions.value(context);
		final int seed = this.randomSeed.value(context);

		final double time = mediaTime.toSecond() * wigglesPerSecond + temporalPhase;

		int ordinal = getBase().ordinal();
		int total = totals[ordinal];
		final double[][] wiggles = new double[total][3];

		for (int i = 0; i < wiggles.length; ++i) {
			// noiseが返す値は-1から1の範囲だが
			// -0.7から0.7の外側はほとんど返らないので、
			// 2倍してソフトクランプしている。

			double spatial = (i+0.5+total*spatialPhase)*(1-correlation);
			if (lockDimensions) {
				wiggles[i][0] = wiggles[i][1] = wiggles[i][2] =
								amount(noise(seed, time, spatial, 113), maxAmount, minAmount);
			} else {
				wiggles[i][0] = amount(noise(seed, time, spatial, 137), maxAmount, minAmount);
				wiggles[i][1] = amount(noise(seed, time, spatial, 73), maxAmount, minAmount);
				wiggles[i][2] = amount(noise(seed, time, spatial, 227), maxAmount, minAmount);
			}
		}

		return new AbstractEvaluator(totals) {
			@Override
			protected double[] evaluate(double[] combine, int index0, int index1, int total) {
				double[] tvec;
				if (index1 == index0 + 1) {
					tvec = wiggles[index0].clone();
				} else {
					double spatial = ((index0+index1)*0.5+total*spatialPhase)*(1-correlation);
					if (lockDimensions) {
						double t = amount(noise(seed, time, spatial, 113), maxAmount, minAmount);
						tvec = new double[] { t, t, t };
					} else {
						tvec = new double[] {
								amount(noise(seed, time, spatial, 137), maxAmount, minAmount),
								amount(noise(seed, time, spatial, 73), maxAmount, minAmount),
								amount(noise(seed, time, spatial, 227), maxAmount, minAmount)
						};
					}
				}
				return applyMode(mode, combine, tvec);
			}
		};
	}

	private double noise(int seed, double arg0, double arg1, double arg2) {
		// arg0,arg1,arg2は整数の場合が多いが、
		// ImprovedNoiseは整数を引数にすると規則的な値を返す場合がある。
		// そのため適当な小数値を足している。
		return ImprovedNoise.noise(
				arg0 + seed*367 + 0.2356,
				arg1 + seed*197 + 0.6049,
				arg2 + seed*271 + 0.7724);
	}

	private double amount(double t, double maxAmount, double minAmount) {
		double max = Math.max(maxAmount, minAmount);
		double min = Math.min(maxAmount, minAmount);
		return (t+1)*0.5*(max-min)+min;
	}

}
