/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;

import org.scannotation.AnnotationDB;

import ch.kuramo.javie.api.GeometryInputType;
import ch.kuramo.javie.api.GeometryOutputType;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.annotations.GeometryShaderParameters;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.ShaderRegistry;
import ch.kuramo.javie.core.services.ShaderRegistryException;

import com.google.inject.Inject;
import com.google.inject.Injector;

public class ShaderRegistryImpl implements ShaderRegistry, IShaderRegistry {

	private final Map<String, ShaderEntry> _entries = Util.newMap();

	@Inject
	private Injector _injector;

	@Inject
	private RenderContext _context;


	private void putEntry(String name, ShaderEntry entry) throws ShaderRegistryException {
		synchronized (_entries) {
			if (_entries.containsKey(name)) {
				throw new ShaderRegistryException(name, "Same name shader already exists.");
			}
			_entries.put(name, entry);
		}
	}

	private ShaderEntry getEntry(String name) {
		synchronized (_entries) {
			return _entries.get(name);
		}
	}

	private ShaderEntry getOrCreateEntry(String name, ShaderType type, String[] attach, String[] source,
							GeometryInputType inputType, GeometryOutputType outputType, int verticesOut) {
		synchronized (_entries) {
			ShaderEntry entry = getEntry(name);
			if (entry == null) {
				source = source.clone();
				for (int i = 0; i < source.length; ++i) {
					if (!source[i].endsWith("\n")) {
						source[i] += "\n";
					}
				}

				entry = new ShaderEntry(name, type, attach, source, inputType, outputType, verticesOut);
				try {
					putEntry(name, entry);
				} catch (ShaderRegistryException e) {
					throw new JavieRuntimeException(e);
				}
			}
			return entry;
		}
	}

	private String fqfn(Class<?> clazz, String name) {
		return clazz.getName() + "." + name;
	}

	private String fqfn(Field field) {
		return fqfn(field.getDeclaringClass(), field.getName());
	}

	public IShaderProgram getProgram(String name) {
		ShaderEntry entry = getEntry(name);
		return (entry != null) ? entry.shaderProgram : null;
	}

	public IShaderProgram getProgram(Class<?> clazz, String name) {
		return getProgram(fqfn(clazz, name));
	}

	public IShaderProgram registerProgram(String name, ShaderType type, String[] attach, String[] source) {
		if (attach == null) {
			attach = new String[0];
		}
		return getOrCreateEntry(name, type, attach, source, null, null, 0).shaderProgram;
	}

	public IShaderProgram registerProgramWithGeometryShader(
			String name, String[] attach, String[] source,
			GeometryInputType inputType, GeometryOutputType outputType, int verticesOut) {

		if (inputType == null || outputType == null || verticesOut <= 0) {
			throw new IllegalArgumentException();
		}
		if (attach == null) {
			attach = new String[0];
		}
		return getOrCreateEntry(name, ShaderType.GEOMETRY_SHADER, attach, source,
								inputType, outputType, verticesOut).shaderProgram;
	}

	public boolean isShaderRegistered(String name) {
		return (getEntry(name) != null);
	}

	public void registerShader(String name, ShaderType type, String[] source) {
		getOrCreateEntry(name, type, null, source, null, null, 0);
	}

	public void registerGeometryShader(String name, String[] source,
			GeometryInputType inputType, GeometryOutputType outputType, int verticesOut) {

		if (inputType == null || outputType == null || verticesOut <= 0) {
			throw new IllegalArgumentException();
		}
		getOrCreateEntry(name, ShaderType.GEOMETRY_SHADER, null, source, inputType, outputType, verticesOut);
	}

	public void scanShaders(AnnotationDB db, ClassLoader cl) throws ShaderRegistryException {
		Set<String> classNames = new HashSet<String>();
		Util.addAll(db.getAnnotationIndex().get(ShaderSource.class.getName()), classNames);

		for (String className : classNames) {
			Class<?> clazz;
			try {
				clazz = Class.forName(className, true, cl);
			} catch (ClassNotFoundException e) {
				// 引数の AnnotationDB と ClassLoader の対応が正しければ必ず見つかるはず。
				throw new IllegalArgumentException(e);
			}

			Object obj = null;

			for (Field field : clazz.getFields()) {
				ShaderSource sourceAnno = null;
				GeometryShaderParameters gspAnno = null;
				for (Annotation fieldAnno : field.getAnnotations()) {
					if (fieldAnno instanceof ShaderSource) {
						sourceAnno = (ShaderSource) fieldAnno;
					} else if (fieldAnno instanceof GeometryShaderParameters) {
						gspAnno = (GeometryShaderParameters) fieldAnno;
					}
				}

				if (sourceAnno == null) {
					continue;
				}

				String[] source;

				if (Modifier.isStatic(field.getModifiers())) {
					source = getSource(null, field);

				} else {
					if (obj == null) {
						try {
							obj = _injector.getInstance(clazz);
						} catch (Exception e) {
							throw new ShaderRegistryException(fqfn(field), e);
						}
					}
					source = getSource(obj, field);
				}

				String[] attach = null;
				if (sourceAnno.program()) {
					attach = sourceAnno.attach().clone();
					for (int i = 0; i < attach.length; ++i) {
						if (attach[i].indexOf('.') == -1) {
							attach[i] = fqfn(field.getDeclaringClass(), attach[i]);
						}
					}
				} else if (sourceAnno.attach().length > 0) {
					throw new ShaderRegistryException(fqfn(field), "'attach' must be empty when 'program' is false");
				}

				String name = fqfn(field);
				ShaderEntry entry;

				if (sourceAnno.type() == ShaderType.GEOMETRY_SHADER && gspAnno != null) {
					if (gspAnno.verticesOut() <= 0) {
						throw new ShaderRegistryException(fqfn(field), "'verticesOut' must be greater than 0");
					}
					entry = new ShaderEntry(name, sourceAnno.type(), attach, source,
									gspAnno.inputType(), gspAnno.outputType(), gspAnno.verticesOut());

				} else {
					entry = new ShaderEntry(name, sourceAnno.type(), attach, source, null, null, 0);
				}

				putEntry(name, entry);
			}
		}
	}

	private String[] getSource(Object obj, Field field) throws ShaderRegistryException {
		Object fieldValue;
		try {
			fieldValue = field.get(obj);
		} catch (IllegalAccessException e) {
			throw new ShaderRegistryException(fqfn(field), e);
		}

		String[] source;

		Class<?> fieldType = field.getType();

		if (fieldType == String[].class) {
			source = (String[]) fieldValue;

		} else if (fieldType == String.class) {

			InputStream in = field.getDeclaringClass().getResourceAsStream((String) fieldValue);
			if (in == null) {
				throw new ShaderRegistryException(fqfn(field), "no such resource found: " + fieldValue);
			}

			try {
				List<String> lines = new ArrayList<String>();
				BufferedReader br = new BufferedReader(new InputStreamReader(in, "UTF-8"));
				try {
					String line;
					while ((line = br.readLine()) != null) {
						lines.add(line);
					}
				} finally {
					br.close();
				}

				source = lines.toArray(new String[lines.size()]);

			} catch (IOException e) {
				throw new ShaderRegistryException(fqfn(field), e);
			}

		} else {
			throw new ShaderRegistryException(fqfn(field),
					"Only String or String[] is allowed to the field type of ShaderSource.");
		}

		if (source == null || source.length == 0) {
			throw new ShaderRegistryException(fqfn(field), "Source is null or empty.");
		}

		for (int i = 0; i < source.length; ++i) {
			if (!source[i].endsWith("\n")) {
				source[i] += "\n";
			}
		}

		return source;
	}

	private void build(GL2 gl, ShaderEntry entry) throws ShaderRegistryException {
		if (entry.shader != 0) {
			throw new IllegalArgumentException("the entry is already built");
		}

		final int type;
		switch (entry.shaderType) {
			case VERTEX_SHADER:
				type = GL2.GL_VERTEX_SHADER;
				break;
			case GEOMETRY_SHADER:
				type = GL2.GL_GEOMETRY_SHADER_ARB;
				break;
			case FRAGMENT_SHADER:
				type = GL2.GL_FRAGMENT_SHADER;
				break;
			default:
				throw new IllegalArgumentException("unsupported ShaderType: " + entry.shaderType);
		}

		int shader = compile(gl, type, entry.source, entry.name);

		if (entry.attach == null) {
			entry.built(shader, 0, null, null);
			return;
		}

		int program = 0;
		try {
			program = gl.glCreateProgram();
			if (program == 0) {
				throw new ShaderRegistryException(entry.name, "glCreateProgram failed.");
			}

			gl.glAttachShader(program, shader);

			for (String attach : entry.attach) {
				ShaderEntry attachEntry = getEntry(attach);
				if (attachEntry == null) {
					throw new ShaderRegistryException(entry.name, "no such shader found: " + attach);
				}
	
				synchronized (attachEntry) {
					if (attachEntry.shader == 0) {
						build(gl, attachEntry);
					}
				}

				gl.glAttachShader(program, attachEntry.shader);

				if (attachEntry.shaderType == ShaderType.GEOMETRY_SHADER && attachEntry.geometryVerticesOut > 0) {
					gl.glProgramParameteri(program, GL2.GL_GEOMETRY_INPUT_TYPE_ARB, attachEntry.geometryInputType.glInputType);
					gl.glProgramParameteri(program, GL2.GL_GEOMETRY_OUTPUT_TYPE_ARB, attachEntry.geometryOutputType.glOutputType);
					gl.glProgramParameteri(program, GL2.GL_GEOMETRY_VERTICES_OUT_ARB, attachEntry.geometryVerticesOut);
				}
			}

			if (entry.shaderType == ShaderType.GEOMETRY_SHADER && entry.geometryVerticesOut > 0) {
				gl.glProgramParameteri(program, GL2.GL_GEOMETRY_INPUT_TYPE_ARB, entry.geometryInputType.glInputType);
				gl.glProgramParameteri(program, GL2.GL_GEOMETRY_OUTPUT_TYPE_ARB, entry.geometryOutputType.glOutputType);
				gl.glProgramParameteri(program, GL2.GL_GEOMETRY_VERTICES_OUT_ARB, entry.geometryVerticesOut);
			}

			gl.glLinkProgram(program);

			int[] status = new int[1];
			gl.glGetProgramiv(program, GL2.GL_LINK_STATUS, status, 0);
			if (status[0] == GL2.GL_TRUE) {
				entry.built(shader, program, getAttributes(gl, program, entry.name), getUniforms(gl, program, entry.name));
				shader = 0;
				program = 0;
				return;
			}

			int[] length = new int[1];
			gl.glGetProgramiv(program, GL2.GL_INFO_LOG_LENGTH, length, 0);

			String infoLog;
			if (length[0] > 0) {
				byte[] infoLogBuf = new byte[length[0]];
				gl.glGetProgramInfoLog(program, length[0], length, 0, infoLogBuf, 0);
				try {
					infoLog = new String(infoLogBuf, 0, length[0], "ISO-8859-1");
				} catch (UnsupportedEncodingException e) {
					infoLog = new String(infoLogBuf, 0, length[0]);
				}
			} else {
				infoLog = "(no info log)";
			}
			throw new ShaderRegistryException(entry.name, "link failed: " + infoLog);

		} finally {
			if (program != 0) gl.glDeleteProgram(program);
			if (shader != 0) gl.glDeleteShader(shader);
		}
	}

	private int compile(GL2 gl, int type, String[] source, String shaderName) throws ShaderRegistryException {
		int shader = gl.glCreateShader(type);
		if (shader == 0) {
			throw new ShaderRegistryException(shaderName, "glCreateShader failed.");
		}

		gl.glShaderSource(shader, source.length, source, null, 0);
		gl.glCompileShader(shader);

		int[] status = new int[1];
		gl.glGetShaderiv(shader, GL2.GL_COMPILE_STATUS, status, 0);
		if (status[0] == GL2.GL_TRUE) {
			return shader;
		}

		int[] length = new int[1];
		gl.glGetShaderiv(shader, GL2.GL_INFO_LOG_LENGTH, length, 0);

		byte[] infoLogBuf = new byte[length[0]];
		gl.glGetShaderInfoLog(shader, length[0], length, 0, infoLogBuf, 0);

		gl.glDeleteShader(shader);

		String infoLog;
		try {
			infoLog = new String(infoLogBuf, 0, length[0], "ISO-8859-1");
		} catch (UnsupportedEncodingException e) {
			infoLog = new String(infoLogBuf, 0, length[0]);
		}
		throw new ShaderRegistryException(shaderName, "compile failed: " + infoLog);
	}

	private Map<String, Attribute> getAttributes(GL2 gl, int program, String programName) throws ShaderRegistryException {
		Map<String, Attribute> map = new HashMap<String, Attribute>();

		int[] int3 = new int[3];

		gl.glGetProgramiv(program, GL2.GL_ACTIVE_ATTRIBUTES, int3, 0);
		int activeAttributes = int3[0];

		gl.glGetProgramiv(program, GL2.GL_ACTIVE_ATTRIBUTE_MAX_LENGTH, int3, 0);
		int bufSize = int3[0];
		byte[] nameBuf = new byte[bufSize];

		for (int i = 0; i < activeAttributes; ++i) {
			// int3[0] : length
			// int3[1] : size
			// int3[2] : type
			gl.glGetActiveAttrib(program, i, bufSize, int3, 0, int3, 1, int3, 2, nameBuf, 0);

			int error = gl.glGetError();
			if (error != GL2.GL_NO_ERROR) {
				throw new ShaderRegistryException(programName,
						"glGetActiveAttrib failed: " + GLU.createGLU(gl).gluErrorString(error));
			}

			String name;
			try {
				name = new String(nameBuf, 0, int3[0], "ISO-8859-1");
			} catch (UnsupportedEncodingException e) {
				name = new String(nameBuf, 0, int3[0]);
			}

			int location = gl.glGetAttribLocation(program, name);

			map.put(name, new Attribute(location, int3[1], int3[2]));
		}

		return map;
	}

	private Map<String, Uniform> getUniforms(GL2 gl, int program, String programName) throws ShaderRegistryException {
		Map<String, Uniform> map = new HashMap<String, Uniform>();

		int[] int3 = new int[3];

		gl.glGetProgramiv(program, GL2.GL_ACTIVE_UNIFORMS, int3, 0);
		int activeUniforms = int3[0];

		gl.glGetProgramiv(program, GL2.GL_ACTIVE_UNIFORM_MAX_LENGTH, int3, 0);
		int bufSize = int3[0];
		byte[] nameBuf = new byte[bufSize];

		for (int i = 0; i < activeUniforms; ++i) {
			// int3[0] : length
			// int3[1] : size
			// int3[2] : type
			gl.glGetActiveUniform(program, i, bufSize, int3, 0, int3, 1, int3, 2, nameBuf, 0);

			int error = gl.glGetError();
			if (error != GL2.GL_NO_ERROR) {
				throw new ShaderRegistryException(programName,
						"glGetActiveUniform failed: " + GLU.createGLU(gl).gluErrorString(error));
			}

			String name;
			try {
				name = new String(nameBuf, 0, int3[0], "ISO-8859-1");
			} catch (UnsupportedEncodingException e) {
				name = new String(nameBuf, 0, int3[0]);
			}

			int location = gl.glGetUniformLocation(program, name);

			map.put(name, new Uniform(location, int3[1], int3[2]));
		}

		return map;
	}


	private class ShaderEntry {
		private final String name;
		private final ShaderType shaderType;
		private String[] attach;
		private String[] source;
		private final GeometryInputType geometryInputType;
		private final GeometryOutputType geometryOutputType;
		private final int geometryVerticesOut;
		private volatile int shader;
		private volatile int program;
		private Map<String, Attribute> attributes;
		private Map<String, Uniform> uniforms;
		private final IShaderProgram shaderProgram;

		private ShaderEntry(
				String name, ShaderType shaderType, String[] attach, String[] source,
				GeometryInputType geometryInputType, GeometryOutputType geometryOutputType, int geometryVerticesOut) {
			this.name = name;
			this.shaderType = shaderType;
			this.attach = attach;
			this.source = source;
			this.geometryInputType = geometryInputType;
			this.geometryOutputType = geometryOutputType;
			this.geometryVerticesOut = geometryVerticesOut;
			this.shaderProgram = (attach != null) ? createShaderProgramImpl() : null;
		}

		private void built(int shader, int program, Map<String, Attribute> attributes, Map<String, Uniform> uniforms) {
			this.shader = shader;
			this.program = program;
			this.attributes = attributes;
			this.uniforms = uniforms;
			attach = null;
			source = null;
		}

		private IShaderProgram createShaderProgramImpl() {
			return new IShaderProgram() {
				public String getName() {
					return name;
				}

				public void useProgram(Runnable r) {
					GL2 gl = _context.getGL().getGL2();

					synchronized (ShaderEntry.this) {
						if (program == 0) {
							try {
								build(gl, ShaderEntry.this);
							} catch (ShaderRegistryException e) {
								throw new JavieRuntimeException(e);
							}
						}

						try {
							gl.glUseProgram(program);
							r.run();
							gl.glFinish();
						} finally {
							gl.glUseProgram(0);
						}
					}
				}

				public int getProgram() {
					return program;
				}

				public int getAttributeLocation(String attributeName) {
					if (program == 0) {
						throw new IllegalStateException();
					}

					Attribute attribute = attributes.get(attributeName);
					if (attribute == null && attributeName.endsWith("[0]")) {
						// TODO Uniformと同じか未確認 -> nVidiaの場合 glGetActiveAttrib が返すattribute名に[0]が付いていない。
						attribute = attributes.get(attributeName.substring(0, attributeName.length()-3));
					}
					return (attribute != null) ? attribute.location : -1;
				}

				public int getUniformLocation(String uniformName) {
					if (program == 0) {
						throw new IllegalStateException();
					}

					Uniform uniform = uniforms.get(uniformName);
					if (uniform == null && uniformName.endsWith("[0]")) {
						// nVidiaの場合 glGetActiveUniform が返すuniform名に[0]が付いていない。
						uniform = uniforms.get(uniformName.substring(0, uniformName.length()-3));
					}
					return (uniform != null) ? uniform.location : -1;
				}
			};
		}
	}

	private static class Attribute {
		private final int location;
//		private final int size;
//		private final int type;

		private Attribute(int location, int size, int type) {
			this.location = location;
//			this.size = size;
//			this.type = type;
		}
	}

	private static class Uniform {
		private final int location;
//		private final int size;
//		private final int type;

		private Uniform(int location, int size, int type) {
			this.location = location;
//			this.size = size;
//			this.type = type;
		}
	}

}
