/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.nio.FloatBuffer;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.services.ITexture1DSupport;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class Texture1DSupportImpl implements ITexture1DSupport {

	private static final float[] FLOAT0000 = new float[] { 0, 0, 0, 0 };

	private final RenderContext context;

	@Inject
	public Texture1DSupportImpl(RenderContext context) {
		this.context = context;
	}

	public int texture1DFromArray(IArray<float[]> array,
			int format, int internalFormat) {

		return texture1DFromArray(array.getArray(), 0, array.getLength(),
				format, internalFormat, GL2.GL_LINEAR, GL2.GL_CLAMP_TO_EDGE);
	}

	public int texture1DFromArray(float[] array,
			int format, int internalFormat) {

		return texture1DFromArray(array, 0, array.length,
				format, internalFormat, GL2.GL_LINEAR, GL2.GL_CLAMP_TO_EDGE);
	}

	public int texture1DFromArray(IArray<float[]> array,
			int format, int internalFormat, int filter, int wrapMode) {

		return texture1DFromArray(array.getArray(), 0, array.getLength(),
				format, internalFormat, filter, wrapMode);
	}

	public int texture1DFromArray(float[] array,
			int format, int internalFormat, int filter, int wrapMode) {

		return texture1DFromArray(array, 0, array.length,
				format, internalFormat, filter, wrapMode);
	}

	public int texture1DFromArray(float[] array, int offset, int length,
			int format, int internalFormat, int filter, int wrapMode) {

		int width;

		switch (format) {
			case GL2.GL_ALPHA:
				width = length;
				break;

			case GL2.GL_RGBA:
				width = length / 4;
				if (width * 4 != length) {
					throw new IllegalArgumentException("length must be multiple of 4");
				}
				break;

			default:
				throw new IllegalArgumentException("format must be GL_FLOAT or GL_RGBA");
		}

		GL2 gl = context.getGL().getGL2();

		int[] texture = new int[1];
		int[] current = new int[1];
		gl.glGetIntegerv(GL2.GL_TEXTURE_BINDING_1D, current, 0);
		try {
			gl.glGenTextures(1, texture, 0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, texture[0]);

			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MIN_FILTER,
					(filter == GL2.GL_NEAREST) ? GL2.GL_NEAREST : GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MAG_FILTER,
					(filter == GL2.GL_NEAREST) ? GL2.GL_NEAREST : GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_S, wrapMode);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_EDGE);

			if (wrapMode == GL2.GL_CLAMP_TO_BORDER) {
				gl.glTexParameterfv(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_BORDER_COLOR, FLOAT0000, 0);
			}

			gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, internalFormat,
					width, 0, format, GL2.GL_FLOAT, FloatBuffer.wrap(array, offset, length));

			if (filter == GL2.GL_LINEAR_MIPMAP_NEAREST || filter == GL2.GL_LINEAR_MIPMAP_LINEAR) {
				gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MIN_FILTER, filter);
				gl.glTexParameterf(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MAX_ANISOTROPY_EXT, 16.0f);
				gl.glGenerateMipmap(GL2.GL_TEXTURE_1D);
			}

			int result = texture[0];
			texture[0] = 0;
			return result;

		} finally {
			gl.glBindTexture(GL2.GL_TEXTURE_1D, current[0]);
			if (texture[0] != 0) gl.glDeleteTextures(1, texture, 0);
		}
	}

}
