/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "stdafx.h"
#include "AviSynthSourceJNI.h"
#include <avisynth.h>


class PrimitiveArrayCritical
{
public:
	PrimitiveArrayCritical(JNIEnv* env, jarray array) : mEnv(env), mArray(array), mBuffer(NULL) { }
	~PrimitiveArrayCritical() { Release(); }

	void* Get()
	{
		if (mBuffer == NULL) {
			mBuffer = mEnv->GetPrimitiveArrayCritical(mArray, NULL);
		}
		return mBuffer;
	}

	void Release()
	{
		if (mBuffer != NULL) {
			mEnv->ReleasePrimitiveArrayCritical(mArray, mBuffer, 0);
		}
	}

//	operator void* ()
//	{
//		return Get();
//	}

private:
	JNIEnv*	mEnv;
	jarray	mArray;
	void*	mBuffer;
};


JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM* vm, void* reserved)
{
	return JNI_VERSION_1_4;
}

JNIEXPORT jlong JNICALL Java_ch_kuramo_javie_core_internal_AviSynthSource_createScriptEnvironment
  (JNIEnv* env, jobject jthis)
{
	try {
		IScriptEnvironment* scriptEnv = CreateScriptEnvironment();
		scriptEnv->SetMemoryMax(32);
		return reinterpret_cast<jlong>(scriptEnv);
	} catch (...) {
		return NULL;
	}
}

JNIEXPORT void JNICALL Java_ch_kuramo_javie_core_internal_AviSynthSource_deleteScriptEnvironment
  (JNIEnv* env, jobject jthis, jlong envPointer)
{
	try {
		delete reinterpret_cast<IScriptEnvironment*>(envPointer);
	} catch (...) {
	}
}

JNIEXPORT jlongArray JNICALL Java_ch_kuramo_javie_core_internal_AviSynthSource_openAVS
  (JNIEnv* env, jobject jthis, jlong envPointer, jbyteArray filename)
{
	try {
		IScriptEnvironment* scriptEnv = reinterpret_cast<IScriptEnvironment*>(envPointer);
		PrimitiveArrayCritical filenameCrit(env, filename);

		AVSValue arg0 = (char*)filenameCrit.Get();
		AVSValue args = AVSValue(&arg0, 1);
		AVSValue imported = scriptEnv->Invoke("Import", args);
		if (!imported.IsClip()) {
			return NULL;
		}
		PClip clip = imported.AsClip();
		VideoInfo vi = clip->GetVideoInfo();

		if (vi.HasVideo() && !vi.IsRGB32()) {
			args = AVSValue(&imported, 1);
			imported = scriptEnv->Invoke("ConvertToRGB32", args);
			if (!imported.IsClip()) {
				return NULL;
			}
			clip = imported.AsClip();
			vi = clip->GetVideoInfo();
		}

		jlong buf[11] = { reinterpret_cast<jlong>(new PClip(clip)),
							vi.width, vi.height, vi.fps_numerator, vi.fps_denominator, vi.num_frames,
							vi.audio_samples_per_second, vi.BytesPerChannelSample(),
							vi.nchannels, vi.num_audio_samples, vi.IsSampleType(SAMPLE_FLOAT) };
		jlongArray result = env->NewLongArray(11);
		env->SetLongArrayRegion(result, 0, 11, buf);
		return result;

	} catch (...) {
		return NULL;
	}
}

JNIEXPORT void JNICALL Java_ch_kuramo_javie_core_internal_AviSynthSource_closeAVS
  (JNIEnv* env, jobject jthis, jlong clipPointer)
{
	try {
		PClip* clip = reinterpret_cast<PClip*>(clipPointer);
		delete clip;
	} catch (...) {
	}
}

JNIEXPORT jboolean JNICALL Java_ch_kuramo_javie_core_internal_AviSynthSource_getFrame
  (JNIEnv* env, jobject jthis, jlong envPointer, jlong clipPointer,
   jint frameNumber, jbyteArray buffer)
{
	try {
		IScriptEnvironment* scriptEnv = reinterpret_cast<IScriptEnvironment*>(envPointer);
		PClip clip = *reinterpret_cast<PClip*>(clipPointer);

		PVideoFrame frame = clip->GetFrame(frameNumber, scriptEnv);

		PrimitiveArrayCritical bufferCrit(env, buffer);

		scriptEnv->BitBlt((BYTE*)bufferCrit.Get(),
							clip->GetVideoInfo().width*4,
							frame->GetReadPtr(),
							frame->GetPitch(),
							frame->GetRowSize(),
							frame->GetHeight());

		return true;

	} catch (...) {
		return false;
	}
}

JNIEXPORT jboolean JNICALL Java_ch_kuramo_javie_core_internal_AviSynthSource_getAudio
  (JNIEnv* env, jobject jthis, jlong envPointer, jlong clipPointer,
   jlong start, jlong count, jbyteArray buffer, jint offset)
{
	try {
		IScriptEnvironment* scriptEnv = reinterpret_cast<IScriptEnvironment*>(envPointer);
		PClip clip = *reinterpret_cast<PClip*>(clipPointer);

		PrimitiveArrayCritical bufferCrit(env, buffer);

		clip->GetAudio((char*)bufferCrit.Get() + offset, start, count, scriptEnv);

		return true;

	} catch (...) {
		return false;
	}
}
