/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.locks.ReentrantLock;

import javax.media.opengl.GL;
import javax.media.opengl.GL2;
import javax.media.opengl.GLCapabilities;
import javax.media.opengl.GLCapabilitiesImmutable;
import javax.media.opengl.GLContext;
import javax.media.opengl.GLDrawableFactory;
import javax.media.opengl.GLException;
import javax.media.opengl.GLPbuffer;
import javax.media.opengl.GLProfile;
import javax.media.opengl.Threading;
import javax.media.opengl.awt.GLCanvas;

import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.GLGlobal;

public class GLGlobalImpl implements GLGlobal {

	private final ReentrantLock _lock = new ReentrantLock();

	private final String _initError;

	private final GLPbuffer _pbuffer;

	private final GLContext _sharedContext;

	private final boolean _macosx;

	private final boolean _windows;

	private final boolean _intel;

	private final boolean _amdAti;

	private final boolean _nvidia;

	private final boolean _radeonHD5k6k;

	private final String _driverVersion;

	private final int _maxSamples;

	private final int _maxTextureImageUnits;

	private final Set<String> _extensions;

	private final String _report;

	private Queue<GLContext> _pool;

	private GLCapabilitiesImmutable _canvasCapabilities;


	public GLGlobalImpl() {
		GLProfile profile = GLProfile.getMaxFixedFunc();

		GLCapabilities glcap = new GLCapabilities(profile);
		glcap.setHardwareAccelerated(true);
		glcap.setDoubleBuffered(false);
		glcap.setOnscreen(true);
		glcap.setPBuffer(false);

		setCanvasCapabilities(glcap);

		glcap.setOnscreen(false);
		glcap.setPBuffer(true);

		GLPbuffer pbuf = null;
		GLException exception = null;
		try {
			pbuf = GLDrawableFactory.getFactory(profile).createGLPbuffer(null, glcap, null, 1, 1, null);
		} catch (GLException e) {
			exception = e;
		}

		final String[] strings = new String[4];
		final int[] integers = new int[2];
		final String[][] extensions = new String[1][];

		if (pbuf != null) {
			_initError = null;
			_pbuffer = pbuf;
			_sharedContext = pbuf.getContext();

			Threading.invokeOnOpenGLThread(new Runnable() {
				public void run() {
					_sharedContext.makeCurrent();
					try {
						GL gl = _sharedContext.getGL();
						strings[0] = gl.glGetString(GL.GL_VENDOR);
						strings[1] = gl.glGetString(GL.GL_VERSION);
						strings[2] = gl.glGetString(GL.GL_RENDERER);
						gl.glGetIntegerv(GL2.GL_MAX_SAMPLES, integers, 0);
						gl.glGetIntegerv(GL2.GL_MAX_TEXTURE_IMAGE_UNITS, integers, 1);

						String extStr = gl.glGetString(GL.GL_EXTENSIONS);
						if (extStr != null) {
							extensions[0] = extStr.split("\\s+");
							Arrays.sort(extensions[0]);
						} else {
							extensions[0] = new String[0];
						}

						strings[strings.length-1] = buildReport(gl, extensions[0]);

					} finally {
						_sharedContext.release();
					}
				}
			});
		} else {
			_initError = exception.getMessage();
			_pbuffer = null;
			_sharedContext = null;
			Arrays.fill(strings, "");
			Arrays.fill(integers, 0);
			extensions[0] = new String[0];
		}

		String osName = System.getProperty("os.name").replaceAll("\\s+", "").toLowerCase();
		_macosx = osName.contains("macosx");
		_windows = osName.contains("windows");

		List<String> lcVendor = Arrays.asList(strings[0].toLowerCase().split("\\s+"));
		_intel = lcVendor.contains("intel");
		_amdAti = lcVendor.contains("amd") || lcVendor.contains("ati");		// ベンダー名に"AMD"を返すものがあるのかどうか知らないが念のため。
		_nvidia = lcVendor.contains("nvidia");
		_radeonHD5k6k = strings[2].toLowerCase().matches(".*radeon hd [56][0-9]{3}.*");

		_driverVersion = diverVersionFromVersionString(strings[1]);

		_maxSamples = integers[0];
		_maxTextureImageUnits = integers[1];

		_extensions = Collections.unmodifiableSet(Util.newLinkedHashSet(Arrays.asList(extensions[0])));
		_report = strings[strings.length-1];
	}

	private String diverVersionFromVersionString(String version) {
		if (_macosx) {
			int index = version.lastIndexOf('-');
			if (index != -1) {
				return version.substring(index+1);
			}
		} else if (_intel) {
			int index = version.lastIndexOf(' ');
			if (index != -1) {
				return version.substring(index+1);
			}
		} else if (_amdAti) {
			int index = version.indexOf(' ');
			if (index != -1) {
				version = version.substring(0, index);
			}
			index = version.lastIndexOf('.');
			if (index != -1) {
				return version.substring(index+1);
			}
		} else if (_nvidia) {
			// NVIDIAではドライババージョンが含まれていない。
		}
		return "Unknown";
	}

	private String buildReport(GL gl, String[] extensions) {
		StringBuilder sb = new StringBuilder();

		sb.append("GL_RENDERER: ").append(gl.glGetString(GL.GL_RENDERER)).append("\n");
		sb.append("GL_VENDOR: ").append(gl.glGetString(GL.GL_VENDOR)).append("\n");
		sb.append("GL_VERSION: ").append(gl.glGetString(GL.GL_VERSION)).append("\n");
		sb.append("GL_SHADING_LANGUAGE_VERSION: ").append(gl.glGetString(GL2.GL_SHADING_LANGUAGE_VERSION)).append("\n");
//		sb.append("Memory: ").append(/* FIXME メモリ量を取得する方法が不明 */).append("\n");

		sb.append("\nExtensions:\n");
		for (String ext : extensions) {
			sb.append(ext).append("\n");
		}

		sb.append("\nMax Values:\n");

		Field[] fields = GL2.class.getFields();
		Arrays.sort(fields, new Comparator<Field>() {
			public int compare(Field f1, Field f2) {
				return f1.getName().compareTo(f2.getName());
			}
		});

		int[] integers = new int[2];
		for (Field f : fields) {
			int mod = f.getModifiers();
			if (Modifier.isStatic(mod) && Modifier.isFinal(mod)
					&& f.getType() == int.class && f.getName().startsWith("GL_MAX_")) {

				try {
					gl.glGetIntegerv((Integer)f.get(null), integers, 0);
					int error = gl.glGetError();
					if (error != GL.GL_NO_ERROR) {
						if (error != GL.GL_INVALID_ENUM) {
							sb.append(f.getName() + ": ERROR=" + error).append("\n");
						}
					} else if (f.getName().equals("GL_MAX_VIEWPORT_DIMS")) {
						sb.append(String.format("GL_MAX_VIEWPORT_DIMS: %d x %d%n", integers[0], integers[1]));
					} else {
						sb.append(f.getName() + ": " + integers[0]).append("\n");
					}
				} catch (Exception e) {
					sb.append(f.getName() + ": ERROR=" + e.getMessage()).append("\n");
				}
			}
		}

		return sb.toString();
	}

	public ReentrantLock getGlobalLock() {
		return _lock;
	}

	public String getInitError() {
		return _initError;
	}

	public GLContext getSharedContext() {
		return _sharedContext;
	}

	public boolean isMacOSX() {
		return _macosx;
	}

	public boolean isWindows() {
		return _windows;
	}

	public boolean isIntel() {
		return _intel;
	}

	public boolean isAmdAti() {
		return _amdAti;
	}

	public boolean isNvidia() {
		return _nvidia;
	}

	public boolean isRadeonHD5k6k() {
		return _radeonHD5k6k;
	}

	public String getDriverVersion() {
		return _driverVersion;
	}

	public int getMaxSamples() {
		return _maxSamples;
	}

	public int getMaxTextureImageUnits() {
		return _maxTextureImageUnits;
	}

	public Set<String> getExtensions() {
		return _extensions;
	}

	public String getReport() {
		return _report;
	}

	public void enterPoolMode(final int poolSize) {
		if (_pool != null) {
			throw new IllegalStateException("already in pool mode");
		}
		if (poolSize <= 0) {
			throw new IllegalArgumentException("poolSize must be one or greater");
		}

		_pool = new ConcurrentLinkedQueue<GLContext>();

		Threading.invokeOnOpenGLThread(new Runnable() {
			public void run() {
				GLCapabilitiesImmutable caps = _pbuffer.getChosenGLCapabilities();
				GLDrawableFactory factory = GLDrawableFactory.getFactory(caps.getGLProfile());
				for (int i = 0; i < poolSize; ++i) {
					GLPbuffer pbuf = factory.createGLPbuffer(null, caps, null, 1, 1, _sharedContext);
					GLContext context = pbuf.getContext();
					context.makeCurrent();
					context.release();
					_pool.offer(context);
				}
			}
		});
	}

	public GLContext createContext() {
		if (_pool != null) {
			GLContext context = _pool.poll();
			if (context == null) {
				throw new JavieRuntimeException("no GLContext is available");
			}
			return context;
		}

		GLCapabilitiesImmutable caps = _pbuffer.getChosenGLCapabilities();
		GLDrawableFactory factory = GLDrawableFactory.getFactory(caps.getGLProfile());
		GLPbuffer pbuf = factory.createGLPbuffer(null, caps, null, 1, 1, _sharedContext);
		GLContext context = pbuf.getContext();
		context.putAttachedObject(this.getClass().getName() + ".pbuffer", pbuf);
		return context;
	}

	public void destroyContext(GLContext context) {
		if (_pool != null) {
			_pool.offer(context);
		} else {
			GLPbuffer pbuf = (GLPbuffer) context.getAttachedObject(this.getClass().getName() + ".pbuffer");
			pbuf.destroy();
		}
	}

	public GLCapabilitiesImmutable getCanvasCapabilities() {
		return _canvasCapabilities;
	}

	public void setCanvasCapabilities(GLCapabilitiesImmutable capabilities) {
		_canvasCapabilities = (GLCapabilitiesImmutable) capabilities.cloneMutable();
	}

	public GLCanvas createCanvas() {
		return new GLCanvas(_canvasCapabilities, null, _sharedContext, null);
	}

}
