/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.output;

import java.io.BufferedOutputStream;
import java.io.DataOutput;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

import javax.media.opengl.glu.GLU;
import javax.vecmath.Matrix4d;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class Javie2MMDOutput extends PushSourceOutput {

	public enum ViewingAngleRounding { TO_NEAREST, DOWN, UP } 


	private double scale = 1;

	private Vec3d offset = Vec3d.ZERO;

	private ViewingAngleRounding viewingAngleRounding = ViewingAngleRounding.TO_NEAREST;

	private OutputStream stream;

	private Camera camera;

	private int writtenFrameCount;

	@Inject
	private RenderContext context;


	public void setScale(double scale) {
		this.scale = scale;
	}

	public void setOffset(Vec3d offset) {
		this.offset = offset;
	}

	public void setViewingAngleRounding(ViewingAngleRounding viewingAngleRounding) {
		this.viewingAngleRounding = viewingAngleRounding;
	}

	@Override
	public void setComposition(Composition composition) {
		if (!(composition instanceof LayerComposition)) {
			throw new UnsupportedOperationException();
		}

		final LayerComposition layerComp = (LayerComposition) composition;

		InvocationHandler handler = new InvocationHandler() {
			public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
				Object result = method.invoke(layerComp, args);
				if (method.getName().equals("renderVideoFrame")) {
					camera = context.getCamera();
				}
				return result;
			}
		};

		super.setComposition((LayerComposition) Proxy.newProxyInstance(
				LayerComposition.class.getClassLoader(),
				new Class[] { LayerComposition.class }, handler));
	}

	@Override
	protected void initialize() {
		// 動画の出力機構を利用してカメラデータを書き出すので、無駄にレンダリングが発生しないようにビデオをオフにする。
		// なお、ここで参照しているコンポジションはコピーなので、内容を変更しても差し支えない。
		LayerComposition comp = (LayerComposition) getComposition();
		for (Layer layer : comp.getLayers()) {
			if (!(layer instanceof CameraLayer) && LayerNature.isVideoNature(layer)) {
				LayerNature.setVideoEnabled(layer, false);
			}
		}

		try {
			stream = new BufferedOutputStream(new FileOutputStream(getFile()));

			// ヘッダ部分はとりあえずゼロで埋めておく。finishメソッドで改めて書き直す。
			for (int i = 0; i < 62; ++i) {
				stream.write(0);
			}
		} catch (IOException e) {
			throw new JavieRuntimeException(e);
		}
	}

	@Override
	protected void finish() {
		if (stream != null) {
			try {
				writeInt(0);		// ライトデータの数
				writeInt(0);		// セルフシャドウデータの数
				stream.close();

				RandomAccessFile out = new RandomAccessFile(getFile(), "rw");
				try {
					String header = "Vocaloid Motion Data 0002\0"
								  + "JKLM"								// MMDから書き出したVMDを見ると"JKLM"と入っているので。
								  + "カメラ・照明\0"
								  + "on Data";							// MMDから書き出したVMDを見ると"on Data"と入っているので。
					out.write(header.getBytes("Shift_JIS"));
					writeInt(out, 0);									// モーションデータの数
					writeInt(out, 0);									// スキンデータの数
					writeInt(out, writtenFrameCount);					// カメラデータの数
				} finally {
					out.close();
				}
			} catch (IOException e) {
				throw new JavieRuntimeException(e);
			}
		}
	}

	@Override
	public boolean isVideoEnabled() {
		return true;
	}

	@Override
	public void setVideoEnabled(boolean videoEnabled) {
		throw new UnsupportedOperationException("can't change videoEnabled");
	}

	@Override
	public boolean isAudioEnabled() {
		return false;
	}

	@Override
	public void setAudioEnabled(boolean audioEnabled) {
		throw new UnsupportedOperationException("can't change audioEnabled");
	}

	@Override
	protected void writeVideo(long frameNumber, Time time, IVideoBuffer vb) {
		double[] prjMatrix = camera.getProjection3D();
		double[] mvMatrix = camera.getModelView3D();

		Size2i size = getComposition().getSize();
		int w = size.width;
		int h = size.height;
		int[] viewport = new int[] { 0, 0, w, h };
		double[][] unprj = new double[4][3];

		GLU glu = context.getGLU();
		glu.gluUnProject(w*0.5, 0, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[0], 0);
		glu.gluUnProject(w*0.5, 0, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[1], 0);
		glu.gluUnProject(w*0.5, h, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[2], 0);
		glu.gluUnProject(w*0.5, h, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[3], 0);

		Vector3d v1 = new Vector3d();
		Vector3d v2 = new Vector3d();
		v1.sub(new Point3d(unprj[1]), new Point3d(unprj[0]));
		v2.sub(new Point3d(unprj[3]), new Point3d(unprj[2]));
		double viewingAngle = Math.acos(v1.dot(v2) / (v1.length() * v2.length()));


		Matrix4d mi = new Matrix4d(mvMatrix);
		mi.transpose();
		mi.invert();

		// カメラの位置
		Point3d pc = new Point3d(0, 0, 0);
		mi.transform(pc);

		// 仮の距離
		double length = -h / (2*Math.tan(viewingAngle/2));

		// 目標点
		Point3d po = new Point3d(0, 0, length);
		mi.transform(po);

		// 距離 (miの変換にスケーリングが含まれている場合は、仮の距離と違う値になる)
		length = -po.distance(pc);

		// 回転Y,X
		Vector3d v = new Vector3d();
		v.sub(po, pc);
		double ry = Math.atan2(v.x, v.z);
		double rx = -Math.atan2(v.y, Math.hypot(v.x, v.z));

		// 回転Z
		Matrix4d mry = new Matrix4d();
		Matrix4d mrx = new Matrix4d();
		mry.rotY(ry);
		mrx.rotX(rx);
		Matrix4d mryx = new Matrix4d();
		mryx.mul(mry, mrx);

		Vector3d n1 = new Vector3d(0, 1, 0);
		Vector3d n2 = new Vector3d(0, 1, 0);
		Vector3d n3 = new Vector3d();
		Vector3d n4 = new Vector3d(0, 0, 1);
		mryx.transform(n1);
		mi.transform(n2);
		n3.cross(n1, n2);
		mryx.transform(n4);
		double dot12 = n1.dot(n2);
		double dot34 = n3.dot(n4);
		double len1 = n1.length();
		double len2 = n2.length();
		double rz = Math.acos(Math.min(Math.max(dot12 / (len1 * len2), -1), 1)) * Math.signum(dot34);

		try {
			writeInt((int)frameNumber);
			writeFloat((float)(length * scale));
			writeFloat((float)((po.x - offset.x) * scale));
			writeFloat((float)(-(po.y - offset.y) * scale));
			writeFloat((float)((po.z - offset.z) * scale));
			writeFloat((float)rx);
			writeFloat((float)-ry);
			writeFloat((float)rz);
			for (int i = 0; i < 12; ++i) {
				stream.write(0x14);
				stream.write(0x6b);
			}

			viewingAngle = Math.toDegrees(viewingAngle);
			switch (viewingAngleRounding) {
				case DOWN:	viewingAngle = Math.floor(viewingAngle); break;
				case UP:	viewingAngle = Math.ceil(viewingAngle); break;
				default:	viewingAngle = Math.round(viewingAngle); break;
			}
			writeInt((int)viewingAngle);
			stream.write(0);

		} catch (IOException e) {
			throw new JavieRuntimeException(e);
		}

		++writtenFrameCount;
	}

	@Override
	protected void writeAudio(long frameNumber, Time time, IAudioBuffer ab) {
		throw new UnsupportedOperationException();
	}

	private void writeInt(DataOutput out, int value) throws IOException {
		for (int i = 0; i < 4; ++i) {
			out.write((value >> (i*8)) & 0xff);
		}
	}

	private void writeInt(int value) throws IOException {
		for (int i = 0; i < 4; ++i) {
			stream.write((value >> (i*8)) & 0xff);
		}
	}

	private void writeFloat(float value) throws IOException {
		writeInt(Float.floatToIntBits(value));
	}

}
