/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryUsage;
import java.lang.management.OperatingSystemMXBean;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.LinkedHashSet;

import javax.media.opengl.GLCapabilities;

import org.eclipse.core.commands.operations.IOperationHistory;
import org.eclipse.core.commands.operations.LinearUndoEnforcer;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.application.IWorkbenchConfigurer;
import org.eclipse.ui.application.IWorkbenchWindowConfigurer;
import org.eclipse.ui.application.WorkbenchAdvisor;
import org.eclipse.ui.application.WorkbenchWindowAdvisor;
import org.eclipse.ui.console.ConsolePlugin;
import org.eclipse.ui.console.IConsole;
import org.eclipse.ui.console.IConsoleManager;
import org.eclipse.ui.console.IOConsoleOutputStream;
import org.eclipse.ui.console.MessageConsole;
import org.osgi.framework.Version;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.app.player.GLCanvasFactory;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.GLGlobal;

public class ApplicationWorkbenchAdvisor extends WorkbenchAdvisor {

	private static final Logger logger = LoggerFactory.getLogger(ApplicationWorkbenchAdvisor.class);


	private void createDefaultConsole() {
		MessageConsole console = new MessageConsole("コンソール", null);
		IConsoleManager consoleManager = ConsolePlugin.getDefault().getConsoleManager();
		consoleManager.addConsoles(new IConsole[] { console });

		IOConsoleOutputStream out = console.newOutputStream();
		IOConsoleOutputStream err = console.newOutputStream();

		System.setOut(new PrintStream(out));
		System.setErr(new PrintStream(err));

		Display display = Display.getDefault();
		//console.setBackground(display.getSystemColor(SWT.COLOR_GRAY));
		err.setColor(display.getSystemColor(SWT.COLOR_RED));
	}

	private boolean checkAndModifyJavieIni() {
		String vmargs = System.getProperty("eclipse.vmargs");
		if (vmargs == null) {
			// eclipse.vmargs が見つからないのは、開発時にEclipseから起動した場合のはず。
			return false;
		}

		Version curVers = Platform.getProduct().getDefiningBundle().getVersion();
		Version iniVers = Version.parseVersion(System.getProperty("javie.ini.version"));

		String extDirs = String.format("%s%s%s%s%s%s%s", System.getProperty("java.home"),
				File.separator, "lib", File.separator, "ext", File.pathSeparator,
				Platform.getOS().equals(Platform.OS_MACOSX) ? "../../../ext" : "ext");
		boolean validExtDirs = extDirs.equals(System.getProperty("java.ext.dirs"));

		for (String arg : vmargs.split("\n|\r\n?")) {
			if (arg.startsWith("-Xmx")) {
				logger.info("-Xmx found in vmargs: " + arg);
				if (iniVers.compareTo(new Version(0, 5, 1)) < 0) {
					return updateJavieIni(0, extDirs, curVers.toString(), iniVers.toString());
				} else if (!validExtDirs) {
					return updateJavieIni(-1, extDirs, curVers.toString(), iniVers.toString());
				}
				return false;
			}
		}

		MemoryMXBean mbean = ManagementFactory.getMemoryMXBean();
		MemoryUsage heapUsage = mbean.getHeapMemoryUsage();
		long curMaxHeap = heapUsage.getMax();

		logger.info("Max Heap Size: " + curMaxHeap);

		long physMemSize;
		try {
			OperatingSystemMXBean osMXBean = ManagementFactory.getOperatingSystemMXBean();
			Class<?> clazz = Class.forName("com.sun.management.OperatingSystemMXBean");
			if (clazz.isInstance(osMXBean)) {
				Method m = clazz.getMethod("getTotalPhysicalMemorySize");
				physMemSize = (Long)m.invoke(osMXBean);
				logger.info("Physical Memory Size: " + physMemSize);
			} else {
				logger.warn("cannot get physical memory size: OperatingSystemMXBean is not instance of com.sun.management.OperatingSystemMXBean");
				return validExtDirs ? false : updateJavieIni(-1, extDirs, curVers.toString(), iniVers.toString());
			}
		} catch (Exception e) {
			logger.warn("error getting physical memory size", e);
			return validExtDirs ? false : updateJavieIni(-1, extDirs, curVers.toString(), iniVers.toString());
		}

		long newMaxHeap;
		if (physMemSize < 256*1000*1000) {
			logger.warn("physical memory size is too little.");
			return validExtDirs ? false : updateJavieIni(-1, extDirs, curVers.toString(), iniVers.toString());
		} else if (physMemSize < 512*1000*1000) {
			newMaxHeap = physMemSize*3/4 - 128*1000*1000;
		} else {
			newMaxHeap = Math.min(physMemSize/2, 512*1000*1000);
		}

		if (newMaxHeap <= curMaxHeap) {
			return validExtDirs ? false : updateJavieIni(-1, extDirs, curVers.toString(), iniVers.toString());
		}

		return updateJavieIni(newMaxHeap, extDirs, curVers.toString(), iniVers.toString());
	}

	private boolean updateJavieIni(long newMaxHeap, String extDirs, String newIniVers, String oldIniVers) {
		newIniVers = "-Djavie.ini.version=" + newIniVers;
		logger.info("Adding to vmargs: " + newIniVers);

		String newXmx;
		if (newMaxHeap > 0) {
			newXmx = "-Xmx" + newMaxHeap;
			logger.info("Adding to vmargs: " + newXmx);
		} else {
			newXmx = null;
		}

		if (extDirs != null) {
			extDirs = "-Djava.ext.dirs=" + extDirs;
			logger.info("Adding to vmargs: " + extDirs);
		}

		try {
			File curIniFile = new File("Javie.ini");
			File newIniFile = new File("Javie.ini.new");
			BufferedReader br = null;
			PrintWriter pr = null;
			try {
				boolean added = false;
				br = new BufferedReader(new FileReader(curIniFile));
				pr = new PrintWriter(newIniFile);
				String line;
				while ((line = br.readLine()) != null) {
					String trim = line.trim();
					if (trim.startsWith("-Djavie.ini.version=")) continue;
					if (trim.startsWith("-Xmx") && newMaxHeap >= 0) continue;
					if (trim.startsWith("-Djava.ext.dirs=")) continue;
					pr.println(line);
					if (trim.equals("-vmargs")) {
						pr.println(newIniVers);
						if (newXmx != null) pr.println(newXmx);
						if (extDirs != null) pr.println(extDirs);
						added = true;
					}
				}
				if (!added) {
					pr.println("-vmargs");
					pr.println(newIniVers);
					if (newXmx != null) pr.println(newXmx);
				}
			} finally {
				if (pr != null) pr.close();
				if (br != null) br.close();
			}
			File oldIniFile = new File(String.format(
					"Javie.ini.%1$s.%2$tY%2$tm%2$td%2$tk%2$tM%2$tS",
					oldIniVers, System.currentTimeMillis()));
			oldIniFile.delete();
			if (!curIniFile.renameTo(oldIniFile)) {
				logger.error("cannot rename Javie.ini to Javie.ini."+oldIniVers);
			}
			if (!newIniFile.renameTo(curIniFile)) {
				logger.error("cannot rename Javie.ini.new to Javie.ini");
				return false;
			}
		} catch (IOException e) {
			logger.error("error modifing Javie.ini", e);
			return false;
		}

		return true;
	}

	public void initialize(final IWorkbenchConfigurer configurer) {
		super.initialize(configurer);

		// このメソッド内で worked ４回
		// Initializer.initialize() 内で３回
		// Eclipse内で１回
		SplashProgressMonitor.beginTask("", 4+3+1);

		// デフォルトコンソールの作成
		SplashProgressMonitor.subTask("Creating Default Console");

		createDefaultConsole();

		SplashProgressMonitor.worked(1);

		// Javie.iniのチェック
		SplashProgressMonitor.subTask("Checking Javie.ini");

		if (checkAndModifyJavieIni()) {
			System.setProperty("javie.relaunch", System.getProperty("eclipse.launcher"));
			configurer.getWorkbench().restart();
			return;
		}

		SplashProgressMonitor.worked(1);


		// その他の初期化
		SplashProgressMonitor.subTask("");

		configurer.setSaveAndRestore(true);

		IOperationHistory history = configurer.getWorkbench().getOperationSupport().getOperationHistory();
		history.addOperationApprover(new LinearUndoEnforcer());

		SplashProgressMonitor.worked(1);


		// OpenGLの初期化
		SplashProgressMonitor.subTask("Initializing OpenGL");

		String os = Platform.getOS();
		boolean win32 = os.equals(Platform.OS_WIN32);
		boolean macosx = os.equals(Platform.OS_MACOSX);

		final GLGlobal glGlobal = InjectorHolder.getInjector().getInstance(GLGlobal.class);
		if (glGlobal.getInitError() == null) {
			logger.info("\n\n***** OpenGL Report *****\n"
					+ "Vendor: " + (glGlobal.isIntel() ? "Intel" :
									glGlobal.isAmdAti() ? "AMD" :
									glGlobal.isNvidia() ? "NVIDIA" : "Unknown") + "\n"
					+ "Driver Version: " + glGlobal.getDriverVersion() + "\n\n"
					+ glGlobal.getReport()
					+ "***** End of OpenGL Report *****\n");

			boolean intel = glGlobal.isIntel();

			if (win32 || (macosx && intel)) {
				// TODO これらの値を環境設定で変更できるようにする。
				int poolSize = (win32 && intel) ? 8 : 20;
				GLCapabilities canvasCap = (GLCapabilities) glGlobal.getCanvasCapabilities().cloneMutable();
				canvasCap.setDoubleBuffered(win32);

				glGlobal.enterPoolMode(poolSize);
				glGlobal.setCanvasCapabilities(canvasCap);

				GLCanvasFactory.getFactory().enterPoolMode(poolSize);
			}
		} else {
			configurer.getWorkbench().getDisplay().asyncExec(new Runnable() {
				public void run() {
					MessageDialog.openError(null,
							"OpenGLの初期化に失敗しました", glGlobal.getInitError());
					configurer.getWorkbench().close();
				}
			});
		}

		SplashProgressMonitor.worked(1);

		Initializer.initialize();
	}

	public boolean preShutdown() {
		// Cmd+Qやメニューから終了しようとすると、
		// ApplicationWorkbenchWindowAdvisor.preWindowShellClose() が呼ばれないまま
		// このメソッドにやってくるようなので、ここでも保存の確認を行う。

		LinkedHashSet<IWorkbenchWindow> windows = Util.newLinkedHashSet();

		// アクティブウインドウが先頭に来るようにする。
		IWorkbench workbench = getWorkbenchConfigurer().getWorkbench();
		IWorkbenchWindow activeWindow = workbench.getActiveWorkbenchWindow();
		if (activeWindow != null) {
			windows.add(activeWindow);
		}

		// 残りのウインドウも表示順にしたいが、ウインドウの表示順を取得する方法が不明なので順不同（たぶん生成順）
		windows.addAll(Arrays.asList(workbench.getWorkbenchWindows()));

		for (IWorkbenchWindow window : windows) {
			ProjectManager pm = ProjectManager.forWorkbenchWindow(window);
			if (pm != null && ApplicationWorkbenchWindowAdvisor.saveConfirmationRequired(window)) {
				switch (SaveUtil.saveIfDirty(pm)) {
					case NOT_DIRTY:
					case SAVED:
					case UNSAVED:
						continue;
					default:
						return false;
				}
			}
		}

		for (IWorkbenchWindow window : windows) {
			Perspective.reset(window);
		}

		return true;
	}

	public WorkbenchWindowAdvisor createWorkbenchWindowAdvisor(IWorkbenchWindowConfigurer configurer) {
		return new ApplicationWorkbenchWindowAdvisor(configurer);
	}

	public String getInitialWindowPerspectiveId() {
		return Perspective.ID;
	}

}
